/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include <QCoreApplication>
#include <QDebug>
#include <math.h>
#include "rf_channel.h"


RF_Channel::RF_Channel(int intrasys_num, int target_channel_num, QString portname)
	: intra_system_number(intrasys_num)
	, target_channel_number(target_channel_num)
	, assigned_port_name(portname)
{

}

/* Destructor function.
 * For safety purposes various things should be disabled when exiting the application.
 * - RF output
 * - RF splitter communication to PAs (ZHL I2C issue)
 * - PSU output */
RF_Channel::~RF_Channel()
{
	qDebug() << "Subsystem[" << intra_system_number << "]: Turning off...";

	bool everything_OFF = true;

	/* Stop the automated readings -> Less stuff on the communication bus. */
	Timer_Readings_Stop(intra_system_number);

	/* Attempt to safely disable everything. */
	for (int i = 0; i < 3; i++)
	{
		everything_OFF = true;

		Set_RF_enable(intra_system_number, false);
		everything_OFF &= (RF_enable == false);

		/* Roughly same thing as Set_PSU_Enable_Safely(), but with a few extra checks... */
		if (config->get_splitter_channel_disable() == true)
		{
			Set_RF_splitter_channels_I2C_enable(intra_system_number, 0x00);		//Does nothing if a splitter board is not configured properly
		}

		if (config->get_PSU_count() > 0 )
		{
			/* Only disable the PSU if the RF splitter channels are disabled or irrelevant
			 * If releveant, but still enabled then don't kill the PSU, as that would cause the I2C bug) */
			if (config->get_splitter_channel_disable() == true && RF_Splitter_channel_I2C_enable == 0x00)
			{
				Set_PSU_Enable(intra_system_number, false);
			}
			else if (config->get_splitter_channel_disable() == false)
			{
				Set_PSU_Enable(intra_system_number, false);
			}
		}

		if (config->get_splitter_channel_disable() == true)
		{
			everything_OFF &= (RF_Splitter_channel_I2C_enable == 0x00);
		}

		if (config->get_PSU_count() > 0)
		{
			everything_OFF &= (PSU_enable_combined == false);
		}

		if (everything_OFF == true)
		{
			break;
		}

		/* If disabling stuff failed -> Cycle the SGx port connection. */
		SGX_port->open();
	}

	if (everything_OFF == false)
	{
		qDebug() << "Subsystem[" << intra_system_number << "]: Failed to turn off correctly!";
	}
	/* Serial Destructor */
	SGX_port->~Serial();
}

void RF_Channel::Initialize(int intrasys_num)
{
	if (intrasys_num != intra_system_number)
	{
		return;
	}

	SGX_port = new Serial();

	/* get the names of the serialports, as well as all the other startup values from config.txt */
	qDebug() << "Subsystem[" << intra_system_number << "]: [Reading config file]";
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	/* Status Checker for error status checking */
	statuschecker = new StatusCheck(SGX_port, QCoreApplication::applicationDirPath() + "/messages.txt");

	/* Error handling */
	timer_reconnect = new QTimer(this);
	connect(SGX_port->serial, &QSerialPort::errorOccurred, this, &RF_Channel::Serial_Error_Handler);
	connect(timer_reconnect, &QTimer::timeout, this, &RF_Channel::Serial_Reconnect);

	/* PSU handling */
	PSU_count = config->get_PSU_count();
	for(int i = 0; i < PSU_count; i++)
	{
		PSU_enable.append(false);
		PSU_voltage.append(0);
		PSU_current.append(0);
		PSU_power.append(0);
	}

	/* Timer for parameter polling */
	timer_readings = new QTimer(this);

//		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_TEST);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_determine_availability_features);

	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_RF_enable);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_power_dbm);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_frequency);

	/* SGx attenuation, magnitude and phase */
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_VGA_attenuation);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_IQMod_magnitude);

	if (config->get_support_phase_control() == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_phase);
	}

	/* NChannel attenuation, magnitude and phase */
	if (config->get_support_splitter_controls_attenuation() == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_VGA_attenuation_nchannel);
	}
	if (config->get_support_splitter_controls_magnitude() == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_IQMod_magnitude_nchannel);
	}
	if (config->get_support_splitter_controls_phase() == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_phase_nchannel);
	}

	// Power Offset
	if (config->get_support_power_offset() == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_offset_db);
	}

#if defined (GUI_4CHANNEL)
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_autophase);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_autophase_enable);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_demodulator_phase);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_demodulator_enable);
#endif
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_SG_error);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_clock_source);
	//Additions for Single Channel System
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_CW_enable);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_DLL_settings);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_DLL_enable);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PWM_settings);
	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_SWP_settings);

	connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_power_control_mode);

	if (config->get_support_PID_control() == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PID_settings);
	}

	//
	// TODO:
	// Use PA type to determine if temperature readouts should be available.
	//

	/* Optionally connect the temperature readout */
	if (config->get_read_temperature() >= 1)
	{
		connect(timer_readings, SIGNAL(timeout()), this, SLOT(poll_temperature_PA()));
	}

	if (config->get_read_temperature_SG() >= 1)
	{
		connect(timer_readings, SIGNAL(timeout()), this, SLOT(poll_temperature_SG()));
	}

	if (config->get_PSU_count() > 0)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PSU_enable);
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PSU_voltage_setpoint);
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PSU_current_limit);
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PSU_dissipation);

		//
		// TODO:
		// Probably shouldn't be #DEBUG reliant.
		// Or the roles should be reversed. It's a bit odd that the 'debug' function is the main one, though it does save a bit of serial traffic...
		//
#ifndef DEBUG
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PSU_IU_All_Readings);
#elif defined DEBUG
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PSU_voltage);
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_PSU_current);
#endif
	}

	SWP_frequency_start = config->get_SWP_start_freq();
	SWP_frequency_stop = config->get_SWP_stop_freq();
	SWP_frequency_step = config->get_SWP_step_freq();
	SWP_power_dbm = convert_watt_to_dbm(config->get_SWP_power_watt());
	SWP_power_watt = config->get_SWP_power_watt();

	external_triggering_synchronization_enable = config->get_external_triggering_synchronization_enable();
	external_triggering_synchronization_delay_us_target = config->get_external_triggering_synchronization_delay_us();
	if (config->get_support_external_triggering_mode())
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_external_triggering_mode);
	}

	/* Activate the Serial communication for the channel */
	SGX_port->setup(assigned_port_name, QString::number(intra_system_number));

	Reconnect_Protocol_Handler();

	if (channel_working == true)
	{
		Initialize_part2();
	}
}

/* Part 2 of initialization procedure; requires functioning communication with the SGx board */
void RF_Channel::Initialize_part2()
{
	/* Overwrite the channel number to the number the channel has been initialized as. */
	Set_Channel_Target(intra_system_number, target_channel_number);
	if (channel_working)
	{
		emit signal_channel_number_get(intra_system_number, channel_number);	//emit that the update has occurred
	}

    /* Get the Version, Identity, HWID and PA Type */
    Get_Firmware_Version();
	Get_Identity();
	Get_Hardware_Identity();
	Get_PA_Type();

    emit signal_firmware_version_get(intra_system_number, channel_number, firmware_version_string);
	emit signal_board_identity_get(intra_system_number, board_manufacturer, board_model, board_serial_number);
	emit signal_hardware_identity_get(intra_system_number, hardware_identity_number, hardware_identity_string);
	emit signal_PA_type_get(intra_system_number, PA_type);

	/* check if the board supports has an ALL chip and DVGA's */
	DVGA_compatible = config->get_support_DVGA();
	ALL_compatible = config->get_menu_ALL_enabled();

	if (ALL_compatible == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_ALL_settings);
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_ALL_enable);
	}

	if (DVGA_compatible == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_DVGA_forward_settings);
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::poll_DVGA_reflected_settings);
	}

	emit signal_ALL_compatible_get(intra_system_number, ALL_compatible);
	emit signal_DVGA_compatible_get(intra_system_number, DVGA_compatible);

	/* Splitter NChannel handling */
	Get_RF_Splitter_Mode();
	if ((splitter_mode % 2) > 0)
	{
		use_substitute_channel_number = true;
	}

	splitter_channel_count = config->get_splitter_channel_count();
	for (int i = 0; i < splitter_channel_count; i++)
	{
		phase_degrees_nchannel.append(0.0);
		magnitude_nchannel.append(0.0);
		attenuation_nchannel.append(0.0);

		PA_power_forward_watt_nchannel.append(0.0);
		PA_power_reflected_watt_nchannel.append(0.0);
		PA_s11_reflection_nchannel.append(0.0);
		PA_power_forward_dbm_nchannel.append(0.0);
		PA_power_reflected_dbm_nchannel.append(0.0);
		PA_s11_nchannel.append(0.0);
	}

	/* Connect Power readings method */
	if (config->get_read_power() == true)
	{
		connect(timer_readings, &QTimer::timeout, this, &RF_Channel::Read_PA_Powers_dBm);

		if (splitter_channel_count >= 2)
		{
			connect(timer_readings, &QTimer::timeout, this, &RF_Channel::Read_PA_Powers_dBm_NChannel);
		}
	}

	/* Share the splitter configuration */
	emit signal_splitter_configuration(intra_system_number, splitter_mode, splitter_channel_count);

	/* Send the initial_error_state */
	emit signal_error_get(intra_system_number, error, statuschecker->translate_SG_error(error));

	/* Reset the SGx board at start-up if configured to do so, otherwise initialization is complete. */
	if (config->get_reset_on_startup())
	{
		channel_init_part2_finished = true;
		channel_init_reset_finished = false;
		Execute_Reset_SGx(intra_system_number);

		/* Remark: Initialization isn't fully finished until the channel has recovered from the reset at startup.
		 * In this situation Serial_Reconnect() will emit the initialization completion signal instead, after succesfully recovering from the reset at startup. */
	}
	else
	{
		channel_init_part2_finished = true;
		channel_init_reset_finished = true;
		emit signal_channel_initialization_complete(intra_system_number, true);
	}
}

void RF_Channel::Timer_Readings_Start(int intrasys_num, int ms)
{
	if (intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Subsystem[" << intra_system_number << "]: Readings started";
	timer_readings->start(ms);
}

void RF_Channel::Timer_Readings_Stop(int intrasys_num)
{
	if (intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Subsystem[" << intra_system_number << "]: Readings stopped";
	timer_readings->stop();
}

/* Remark: RCM Class must disable the error handler for blind RCM
 * This is done by connecting/disconnecting the SerialPortError signal and this slot function as needed. */
void RF_Channel::Serial_Error_Handler(QSerialPort::SerialPortError error)
{
	if (error == QSerialPort::SerialPortError::NoError)
	{
		return;
	}

	if(error == QSerialPort::SerialPortError::TimeoutError)
	{
		/* Ignore serialPort timeout errors */
		return;
	}

	qDebug() << "Subsystem[" << intra_system_number << "]: SerialPortError: " << error;
	Reconnect_Protocol_Handler();
}

/* How the software should handle various reconnect situations */
void RF_Channel::Reconnect_Protocol_Handler()
{
	/* If no port is assigned to the channel; channel initialization is impossible; Halt all channel activity.
	 * + Emit signal that initialization has failed and let the main thread choose how to deal with it.
	 * Else port is defined; Try to reconnect (and finish the init sequence) */

	channel_working = false;
	emit signal_channel_working(intra_system_number, false);

	if (assigned_port_name == "")
	{
		qDebug() << "Subsystem[" << intra_system_number << "]: No port defined in config  -> Could not connect to SGx Board at start -> Stopping Channel.";
		emit signal_channel_initialization_complete(intra_system_number, false);
		Timer_Readings_Stop(intra_system_number);
	}
	else
	{
		timer_reconnect->start(200);
	}
}

/* Try to (re-)open the SGX_port and get the RF Channel back into a usable state */
void RF_Channel::Serial_Reconnect()
{
	if(SGX_port->open())
	{
		if (Serial_Probe() == true)
		{
			channel_working = true;
			Set_Channel_Target(intra_system_number, target_channel_number);
			emit signal_channel_number_get(intra_system_number, channel_number);
			emit signal_channel_working(intra_system_number, channel_working);

			/* If init sequence wasn't completed yet, complete it. */
			if (channel_init_part2_finished == false)
			{
				Initialize_part2();
			}
			else if (channel_init_part2_finished == true && channel_init_reset_finished == false) /* The start-up reset has completed */
			{
				channel_init_reset_finished = true;
				emit signal_channel_initialization_complete(intra_system_number, true);
			}

			timer_reconnect->stop();
		}
	}
}

/* Ensure that the SGx boards haven't swapped serial port names after a reset. Check if the connected SGx board has an expected channel number */
bool RF_Channel::Serial_Probe()
{
	bool ret = false;
	QString rx = "";

	#warning can get stuck here infinitely if an available COM port is specified in config, but no correct device is behind it
	while(!rx.contains("\r\n") && !rx.contains("$IDN,"))
	{
		//
		// TODO: Windows only bug?
		// Channel 2+ appears to have a pretty high chance of being read as empty string on Windows; Issue doesn't appear to occur on Linux/Raspberry.
		// Workaround found: resending the command before reading out again. Proper research would still be appreciated...
		// if this manages to bug out the SGx thread can't shut down.
		//

		rx += SGX_port->writeRead("$IDN,0\r\n");	//Doesn't require channel_working to be true
	}

	int current_channel_number = rx.split(",").at(1).toInt();
	int channelcount = config->get_channel_count();

	for (int i = 0; i < current_channel_number; i++)
	{
//		qDebug() << current_channel_number << intra_system_number + (i * channelcount);
		if (current_channel_number == intra_system_number + (i * channelcount))
		{
			ret = true;
			break;
		}
	}

//	qDebug() << ret;
	return ret;
}


/**********************************************************************************************************************************************************************************
 * Parameter Return functions: return the value of various parameters.
 *********************************************************************************************************************************************************************************/
//MISC
quint64 RF_Channel::Runtime_SGx()
{
	return runtime_SGx;
}
bool RF_Channel::RF_Enable()
{
	return RF_enable;
}
double RF_Channel::Temperature()
{
	return temperature_PA;
}

//CW
double RF_Channel::SGx_Power_dBm()
{
	return SGx_power_dBm;
}
double RF_Channel::VGA_Attenuation()
{
	return VGA_attenuation;
}
double RF_Channel::IQMod_Magnitude()
{
	return IQMod_magnitude;
}
double RF_Channel::Power_dBm()
{
	return power_dbm;
}
double RF_Channel::Power_watt()
{
	return power_watt;
}
double RF_Channel::Frequency()
{
	return frequency_mhz;
}
double RF_Channel::Offset_dB()
{
	return offset_db;
}
double RF_Channel::Phase()
{
	return phase_degrees;
}
double RF_Channel::AutoPhase()
{
	return autophase_degrees;
}
bool RF_Channel::AutoPhase_Enable()
{
	return autophase_enable;
}

double RF_Channel::Phase_NChannel(int subchannel)
{
	return phase_degrees_nchannel[subchannel];
}
double RF_Channel::Attenuation_NChannel(int subchannel)
{
	return attenuation_nchannel[subchannel];
}
double RF_Channel::Magnitude_NChannel(int subchannel)
{
	return magnitude_nchannel[subchannel];
}

//PWM
double RF_Channel::PWM_Duty_Cycle()
{
	return PWM_duty_cycle_target;		//PWM duty cycle is a bit awkward...
}
double RF_Channel::PWM_Frequency()
{
	return PWM_frequency;
}

//DLL
double RF_Channel::DLL_Start_Frequency()
{
	return DLL_start_frequency;
}
double RF_Channel::DLL_Step_Frequency()
{
	return DLL_step_frequency;
}
double RF_Channel::DLL_Frequency_Limit_Lower()
{
	return DLL_frequency_limit_lower;
}
double RF_Channel::DLL_Frequency_Limit_Upper()
{
	return DLL_frequency_limit_upper;
}
double RF_Channel::DLL_Threshold()
{
	return DLL_threshold;
}
double RF_Channel::DLL_Main_Delay()
{
	return DLL_main_delay;
}

//ALL
double RF_Channel::ALL_Threshold()
{
	return ALL_threshold;
}
double RF_Channel::ALL_Frequency_Limit_Lower()
{
	return ALL_frequency_limit_lower;
}
double RF_Channel::ALL_Frequency_Limit_Upper()
{
	return ALL_frequency_limit_upper;
}

//DVGA
double RF_Channel::DVGA_Attenuation_Forward()
{
	return DVGA_attenuation_forward;
}
double RF_Channel::DVGA_Attenuation_Reflected()
{
	return DVGA_attenuation_reflected;
}

//SWP
double RF_Channel::SWP_Start_Frequency()
{
	return SWP_frequency_start;
}
double RF_Channel::SWP_Stop_Frequency()
{
	return SWP_frequency_stop;
}
double RF_Channel::SWP_Step_Frequency()
{
	return SWP_frequency_step;
}
double RF_Channel::SWP_Power_dBm()
{
	return SWP_power_dbm;
}
double RF_Channel::SWP_Power_watt()
{
	return SWP_power_watt;
}
QString RF_Channel::SWP_Raw_Data()
{
	return SWP_raw_data;
}

//PID
double RF_Channel::PID_Kp()
{
	return PID_kp;
}
double RF_Channel::PID_Ki()
{
	return PID_ki;
}
double RF_Channel::PID_Kd()
{
	return PID_kd;
}
double RF_Channel::PID_Offset()
{
	return PID_offset;
}
double RF_Channel::PID_Setpoint()
{
	return PID_setpoint;
}
double RF_Channel::PID_Scaling()
{
	return PID_scaling;
}

//PSU
double RF_Channel::PSU_Voltage_Setpoint()
{
	return PSU_voltage_setpoint;
}
double RF_Channel::PSU_Current_Limit()
{
	return PSU_current_limit;
}


/**********************************************************************************************************************************************************************************
 * RF CHANNEL SERIAL COMMANDS
 *********************************************************************************************************************************************************************************/
/* Convenience feature for verifying string contents of answer from SGx */
bool RF_Channel::checkStringContents(QString inputString, QString string_startsWith, QString string_endsWith, QString string_excludes)
{
	if (inputString.startsWith(string_startsWith) &&
		inputString.endsWith(string_endsWith) &&
		! inputString.contains(string_excludes))
	{
		return true;
	}
	else
	{
		qDebug() << "Subsystem[" << intra_system_number << "]: Communication Fault:" << inputString << "(Expected:" << string_startsWith << string_endsWith << "without" << string_excludes << ")";
		return false;
	}
}

/**********************************************************************************************************************************************************************************
 * Channel Nummber: Set/Get the identifier number of readings.
 *********************************************************************************************************************************************************************************/

void RF_Channel::Set_Channel_Target(int intrasys_num, int chan_num)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	//
	// TODO:
	// Some while looping might be good here to ensure the channel is REALLY set.
	//

	target_channel_number = chan_num;
	Get_Channel();

	if (channel_number != target_channel_number)
	{
		Set_Channel(target_channel_number);
	}

	//
	// TODO:
	// There is some improving to do with the updating of the channel here...
	//
}

/* This SET functions is only for internal use!
 * To set the channel number use Set_Channel_Target instead. */
void RF_Channel::Set_Channel(int chan_num)
{
	if (!channel_working)
	{
		return;
	}

//	QString rx = SGX_port->writeRead("$CHANS," + QString::number(channel_number) + "," + QString::number(chan_num));
	QString rx = SGX_port->writeRead("$CHANS,0," + QString::number(chan_num));

	if (checkStringContents(rx, "$CHANS,", ",OK\r\n", ",ERR"))
	{
		channel_number = chan_num;
		substitute_channel_number = 200 + chan_num;
		qDebug() << "Subsystem[" << intra_system_number << "]: Channel number set to: " << channel_number;
		emit signal_channel_number_get(intra_system_number, channel_number);	//emit that the update has occurred
	}
}

void RF_Channel::Get_Channel()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$CHANG");

	if (checkStringContents(get_string, "$CHANG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		channel_number = QString(get_list.at(1)).toDouble();
	}
}

/**********************************************************************************************************************************************************************************
 * Reset: execute a reset on the SGx board.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Execute_Reset_SGx(int intrasys_num)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Subsystem[" << intra_system_number << "]: Executing SGx Board Reset";
	QString get_string = SGX_port->writeRead("$RST," + QString::number(channel_number));

	checkStringContents(get_string, "$RST,", ",OK\r\n", ",ERR");
}

/**********************************************************************************************************************************************************************************
 * Protections Board Reset: Rearm / reset the protection board.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Execute_Reset_Protection(int intrasys_num)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PROTRST," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PROTRST,", ",OK\r\n", ",ERR"))
	{
		qDebug() << "Subsystem[" << intra_system_number << "]: Executed Protection Board Reset";
	}
}

/**********************************************************************************************************************************************************************************
 * Identity: Get the the type of the SGx board.
 *********************************************************************************************************************************************************************************/
/* Get Manufacturer, Model and Serial number of a board - May be overridden in the EEPROM with $IDNS */
void RF_Channel::Get_Identity()
{
	if (!channel_working)
	{
		return;
	}

	/* Response example: $IDN,1,PinkRF SGx,pRF_SG4A_2450_SMA_DV */
	QString rx = SGX_port->writeRead("$IDN," + QString::number(channel_number));

	if (checkStringContents(rx, "$IDN,", "\r\n", ",ERR"))
	{
		rx.remove("\r");
		rx.remove("\n");

		QStringList get_list = rx.split(",");
		if (get_list.count() >= 3)
		{
			board_manufacturer = get_list.at(2);
		}
		if (get_list.count() >= 4)
		{
			board_model = get_list.at(3);
		}
		if (get_list.count() >= 5)
		{
			board_serial_number = get_list.at(4);
		}

		qDebug() << "Subsystem[" << intra_system_number << "]: Get_Identity ->" << board_manufacturer << board_model << board_serial_number;
	}
}

/* Get the Hardware Identity of the device - Should never change */
void RF_Channel::Get_Hardware_Identity()
{
	if (!channel_working)
	{
		return;
	}

	/*
	 * Response Examples:
	 * $HWIDG,1,0,pRF_SG4A_2450_SMA_DV
	 * $HWIDG,1,12,RFS2G42G5020X+\r\n
	 */

	QString rx = SGX_port->writeRead("$HWIDG," + QString::number(channel_number));

	if (checkStringContents(rx, "$HWIDG,", "\r\n", ",ERR"))
	{
		QStringList get_list = rx.split(",");
		hardware_identity_number = get_list.at(2).toInt();
		hardware_identity_string = get_list.at(3);
		hardware_identity_string.replace("\r", "");
		hardware_identity_string.replace("\n", "");

		qDebug() << "Subsystem[" << intra_system_number << "]: Get_Hardware_Identity ->" << hardware_identity_number << hardware_identity_string;
	}
}


/**********************************************************************************************************************************************************************************
 * PA type: Get the PA type of the Device.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Get_PA_Type()
{
	if (!channel_working)
	{
		return;
	}

	/* Response Example: $PATG,1,8 */
	QString rx = SGX_port->writeRead("$PATG," + QString::number(channel_number));

	if (checkStringContents(rx, "$PATG,", "\r\n", ",ERR"))
	{
		QStringList get_list = rx.split(",");
		PA_type = get_list.at(2).toInt();

		qDebug() << "Subsystem[" << intra_system_number << "]: Get_PA_Type -> " << PA_type;
	}
}


/**********************************************************************************************************************************************************************************
 * RF Splitter: Manage configuration of an RF Splitter
 *********************************************************************************************************************************************************************************/

/* 0 – Ignore RF Splitter
 * 1 – Leijenaar Electronics LE024 splitter  – 4x RF outputs
 * 2 – Leijenaar Electronics LE024 splitter – 1x RF output
 * 3 – Mini-Circuits V2 splitter – 4x RF outputs
 * 4 – Mini-Circuits V2 splitter – 1x RF output  */

/* Set the RF Splitter Mode -
 * Not a Slot */
void RF_Channel::Set_RF_Splitter_Mode(int intrasys_num, int mode)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_RF_Splitter_Mode(" << intra_system_number << "):" << mode;
	QString rx = SGX_port->writeRead("$RSS," + QString::number(channel_number) + "," + QString::number(mode));

	if (checkStringContents(rx, "$RSS,", ",OK\r\n", ",ERR"))
	{
		splitter_mode = mode;
	}
}

/* Get the RF Splitter Mode */
void RF_Channel::Get_RF_Splitter_Mode()
{
	if (!channel_working)
	{
		return;
	}

	QString rx = SGX_port->writeRead("$RSG," + QString::number(channel_number));

	if (checkStringContents(rx, "$RSG,", "\r\n", ",ERR"))
	{
		QStringList get_list = rx.split(",");
		splitter_mode = get_list.at(2).toInt();
	}
}

/* Enable / Disable the I2C connectivity of PA channels behind an RF/I2C splitter board */
void RF_Channel::Set_RF_splitter_channels_I2C_enable(int intrasys_num, int enable_bitmask)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	/* RF splitter / Phase Gain splitter board must be present */
	if (config->get_splitter_channel_disable() == true)
	{
		qDebug() << "Set_RF_splitter_channels_I2C_enable(" << intra_system_number << "):" << enable_bitmask;
		QString rx = SGX_port->writeRead("$MCIES," + QString::number(channel_number) + "," + QString::number(enable_bitmask));

		if (checkStringContents(rx, "$MCIES,", ",OK\r\n", ",ERR"))
		{
			RF_Splitter_channel_I2C_enable = enable_bitmask;
		}
	}
}


/**********************************************************************************************************************************************************************************
 * Auto-gain: Set/Get auto-gain is enabled.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_Autogain_Enable(bool enable)
{
	if (!channel_working)
	{
		return;
	}

	qDebug() << "Autogain enable(" << intra_system_number << "):" << enable;
	QString rx = SGX_port->writeRead("$AGES," + QString::number(channel_number) + "," + QString::number((int)enable));

	if (checkStringContents(rx, "$AGES,", ",OK\r\n", ",ERR"))
	{
		autogain_enable = enable;
	}
}

void RF_Channel::Get_Autogain_Enable()
{
	if (!channel_working)
	{
		return;
	}

	QString rx = SGX_port->writeRead("$AGEG," + QString::number(channel_number));


	if (checkStringContents(rx, "$AGEG,", "\r\n", ",ERR"))
	{
		QStringList get_list = rx.split(",");
		autogain_enable = (bool)(get_list.at(2).toInt());
	}
}


/**********************************************************************************************************************************************************************************
 * Amplifier Mode: Set/Get Amplifier Mode enabled + Set/Get GPIOs if applicable
 *********************************************************************************************************************************************************************************/

#if defined(Q_OS_LINUX)

/* Slot function */
/* RF_system (1channel) periodically sends updates with the GPIO values used to determine whether the SGx is in control or an external source */
void RF_Channel::handler_GPIO_amplifier_mode_get(int intrasys_num, bool GPIO_amplifier_mode)
{
	if (intrasys_num != intra_system_number)
	{
		return;
	}

	GPIO_amplifier_mode_enable = GPIO_amplifier_mode;
}

/* Not a slot */
void RF_Channel::Set_GPIO_Amplifier_Mode_Enable(bool enable)
{
	GPIO_amplifier_mode_enable = enable;
	emit signal_GPIO_amplifier_mode_enabled_set(intra_system_number, GPIO_amplifier_mode_enable);
}

#endif

/* Not a slot */
void RF_Channel::Set_Amplifier_Mode_Enable(bool enable)
{
	switch(config->get_support_amplifier_mode())
	{
		case 1:		//Native Amplifier Mode - SG hardware supports Amplifier mode natively. Uses $RFSS.

			if (enable == true)
			{
				Set_RF_Source(intra_system_number, 1);
			}
			else
			{
				Set_RF_Source(intra_system_number, 0);
			}

			amplifier_mode_enable = (bool)RF_source;	//Amplifier mode reflects the state of the RF Source

			break;

		case 2:		//Bypass Amplifier Mode - External signal bypasses the controller and goes straight to the PA.

			#if defined(Q_OS_LINUX)
			if (config->get_support_HAT_B14_0835())
			{
				Set_GPIO_Amplifier_Mode_Enable(enable);
			}
			#endif

			amplifier_mode_enable = enable;
			break;

		case 3:		//SSG Amplifier Mode - Requires modified SGx board.

			/* Remark: Within RF Channel, if there is no HAT present on the RPi, Bypass mode and SGx mode are the same
			 * However it does make a difference for the implementation on GUI level */

			amplifier_mode_enable = enable;
			break;

		default:
			break;
	}

}

/* Not a slot */
void RF_Channel::Get_Amplifier_Mode_Enable()
{
	switch(config->get_support_amplifier_mode())
	{
		case 1:		//Native Amplifier Mode - SG hardware supports Amplifier mode natively.

			Get_RF_Source();
			amplifier_mode_enable = (bool) RF_source;

			break;

		case 2:		//Bypass Amplifier Mode - External signal bypasses the controller and goes straight to the PA.

			#if defined(Q_OS_LINUX)
			if (config->get_support_HAT_B14_0835())
			{
				amplifier_mode_enable = GPIO_amplifier_mode_enable;
			}
			#endif

			/* If there is no RPi HAT, then there is nothing to read/poll.
			 * Amplifier mode is whatever was last configured by the Set function. */

			break;

		case 3:		//SSG Amplifier Mode - Requires modified SGx board.
			/* There is nothing to read/poll.
			 * Amplifier mode is whatever was last configured by the Set function. */
			break;

		default:

			break;
	}
}

/* Not a Slot */
/* Native Amplifier mode for 50W RFS devices */
void RF_Channel::Set_RF_Source(int intrasys_num, int source)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	/* Usage: $RFSS,channel,source
	 * Source 0 -> Internal VCO
	 * Source 1 -> External RF input */

	QString rx = SGX_port->writeRead("$RFSS," + QString::number(channel_number) + "," + QString::number(source));
	QStringList get_list = rx.split(",");

	if (checkStringContents(rx, "$RFSS,", ",OK\r\n", ",ERR"))
	{
		RF_source = source;
		qDebug() << "Subsystem[" << intra_system_number << "]: Set_RF_Source -> " << RF_source;
	}
}


void RF_Channel::Get_RF_Source()
{
	if (!channel_working)
	{
		return;
	}

	QString rx = SGX_port->writeRead("$RFSG," + QString::number(channel_number));

	if (checkStringContents(rx, "$RFSG,", "\r\n", ",ERR"))
	{
		QStringList get_list = rx.split(",");
		RF_source = get_list.at(2).toInt();
	}
}


/**********************************************************************************************************************************************************************************
 * Power Control Mode: Set Power Control mode / Figure out which power control method is being used.
 *********************************************************************************************************************************************************************************/
//
// TODO:
// Missing firmware AIG command
// Missing means to control PID over RCM / EEPROM
//
/*
	Detect Operational mode:
	- Normal
	  * Autogain ON ($AGEG)
	  * AIS OFF
	  * PID OFF
	- Analog Input Mode
	  * AIS enabled ($AIG missing from command set!!)
	  * Autogain OFF
	  * PID OFF
	- PID Control Mode
	  * PID enabled (Cannot be set over RCM D:)
	  * AIS OFF
	  * Autogain ON?? (But doesn't matter in the 27MHz system cause that one is still controlled from PWRS... fml)
	- Feed FWD Mode
	  * Autogain OFF
	  * AIS OFF
	  * PID OFF
*/

void RF_Channel::Set_Power_Control_Mode(int intrasys_num, int mode)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	switch(mode)
	{
		case POWER_CONTROL_NORMAL:
			Set_Analog_Input_Enable(false);
//			Set_PID_Enable(intra_system_number, false);
			Set_Amplifier_Mode_Enable(false);
			Set_Autogain_Enable(true);
			break;
		case POWER_CONTROL_ANALOG_INPUT:
//			Set_PID_Enable(intra_system_number, false);
			Set_Autogain_Enable(false);
			Set_Amplifier_Mode_Enable(false);
			Set_Analog_Input_Enable(true);			// TODO: Consider flipping the autogain enable state inside this function, to avoid the delayed detection mention above
			break;
		case POWER_CONTROL_FEED_FWD:
			Set_Analog_Input_Enable(false);
//			Set_PID_Enable(intra_system_number, false);
			Set_Amplifier_Mode_Enable(false);
			Set_Autogain_Enable(false);

			//
			// TODO:
			// Workaround below to be resolved and deleted.
			// GCS isn't affected until sending 0 twice over the GUI... for some reason. After that it seems to work just fine.
			//

			if (config->get_FFWD_attenuation_override() >= 0)
			{
				Set_VGA_attenuation(intra_system_number, config->get_FFWD_attenuation_override());
				Set_VGA_attenuation(intra_system_number, config->get_FFWD_attenuation_override());
			}

			break;
		case POWER_CONTROL_AMPLIFIER:
			Set_Autogain_Enable(false);
//			Set_PID_Enable(intra_system_number, false);
			Set_Analog_Input_Enable(false);
			Set_Amplifier_Mode_Enable(true);

			/* Set CW Mode */
			Set_PWM_enable(intra_system_number, false);
			Set_DLL_enable(intra_system_number, false);
			Set_ALL_enable(intra_system_number, false);
			emit signal_CW_enable_get(intra_system_number, !(PWM_enable || DLL_enable || ALL_enable));

			break;
		case POWER_CONTROL_PID:
			Set_Analog_Input_Enable(false);
			Set_Autogain_Enable(true);
			Set_Amplifier_Mode_Enable(false);
//			Set_PID_Enable(intra_system_number, false);
			break;
		default:
			break;
	}

	/* Determine availability of features that are dependent on the power control mode. */
	determine_availability_external_triggering();
	determine_availability_sweep();
	determine_availability_DLL();
	determine_availability_ALL();

	/* Update the power control mode */
	Identify_Power_Control_Mode();
	if (power_control_mode != last_power_control_mode)
	{
		emit signal_power_control_mode_get(intra_system_number, power_control_mode);
	}
	last_power_control_mode = power_control_mode;
}


/* Determine which power control control mode is active */
int RF_Channel::Identify_Power_Control_Mode()
{
	if (autogain_enable == true && analog_input_enable == false && amplifier_mode_enable == false)
	{
		power_control_mode = POWER_CONTROL_NORMAL;
	}
	else if (autogain_enable == false && analog_input_enable == false && amplifier_mode_enable == false)
	{
		power_control_mode = POWER_CONTROL_FEED_FWD;
	}
	else if (autogain_enable == false && analog_input_enable == true && amplifier_mode_enable == false)
	{
		power_control_mode = POWER_CONTROL_ANALOG_INPUT;
	}
	else if (autogain_enable == false && analog_input_enable == false && amplifier_mode_enable == true)
	{
		/* Remark:
		 * Amplifier mode is not always detectable. It depends on the support_amplifier_mode configuration.
		 * - Native Amplifier mode: Can use RFSG to detect amplifier mode.
		 * - Bypass Amplifier mode: Can use GPIOs to detect amplifier mode, though probably nobody will ever touch those...
		 * - SSG Amplifier mode:  No commands available to facilitate detection. It's a GUI induced state only. */
		power_control_mode = POWER_CONTROL_AMPLIFIER;
	}
	else
	{
//		qDebug() << "Subsystem[" << intra_system_number << "]: Power Control Mode conflict -> Auto-gain =" << autogain_enable << ", Analog mode =" << analog_input_enable << ", Amplifier mode =" << amplifier_mode_enable;
		power_control_mode = POWER_CONTROL_UNKNOWN;
	}

	return power_control_mode;
}

/**********************************************************************************************************************************************************************************
 * PA Power readings: get the forward and reflected power of the PA, calculate S11 ratio; emit the readings.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Update_PA_Power(int intrasys_num)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	Read_PA_Powers_interpret_readings();

	if (splitter_channel_count >= 2)
	{
		Read_PA_Powers_interpret_readings_NChannel();
	}
}

void RF_Channel::Read_PA_Powers_dBm()
{
	if (!channel_working)
	{
		return;
	}

	QString rx = "";
	rx = SGX_port->writeRead("$PPDG," + QString::number(channel_number));

	if (checkStringContents(rx, "$PPDG,", "\r\n", ",ERR"))
	{
		QStringList get_list = rx.split(",");

		PA_power_forward_dbm = get_list.at(2).toDouble();
		PA_power_reflected_dbm = get_list.at(3).toDouble();

		/* Subtract dB's to compensate for Duty Cycle */
		if (config->get_duty_cycle_compensation_enabled())
		{
			PA_power_forward_dbm += convert_percent_to_dB(PWM_duty_cycle);
			PA_power_reflected_dbm += convert_percent_to_dB(PWM_duty_cycle);
		}

		PA_power_forward_watt = convert_dbm_to_watt(PA_power_forward_dbm);
		PA_power_reflected_watt = convert_dbm_to_watt(PA_power_reflected_dbm);

		Read_PA_Powers_interpret_readings();
	}
}

void RF_Channel::Read_PA_Powers_interpret_readings()
{
	double FWD_dbm, RFL_dbm, S11, FWD_watt, RFL_watt, S11_reflection;

	Read_PA_Powers_calc_S11_dBm(FWD_dbm, RFL_dbm, S11);
	Read_PA_Powers_calc_S11_watt(FWD_watt, RFL_watt, S11_reflection);
	emit signal_PA_power_readings(intra_system_number, FWD_dbm, RFL_dbm, S11, FWD_watt, RFL_watt, S11_reflection);
}

/* Convert power and S11 values to logarithmic format */
void RF_Channel::Read_PA_Powers_calc_S11_dBm(double &FWD, double &RFL, double &S11)
{
	PA_s11 = PA_power_reflected_dbm - PA_power_forward_dbm;

	FWD = PA_power_forward_dbm;
	RFL = PA_power_reflected_dbm;
	S11 = PA_s11;
}

/* Convert power and S11 values to linear format */
void RF_Channel::Read_PA_Powers_calc_S11_watt(double &FWD, double &RFL, double &S11)
{
	PA_s11_reflection = (PA_power_reflected_watt / PA_power_forward_watt) * 100;

	FWD = PA_power_forward_watt;
	RFL = PA_power_reflected_watt;
	S11 = PA_s11_reflection;
}


/* Alternative PA power Get for a single SGx device controlling multiple PAs.
 * Get the individual FWD + RFL powers of each PA (in Watt) */
void RF_Channel::Read_PA_Powers_NChannel()
{
	if (!channel_working)
	{
		return;
	}

	QString rx = "";
	rx = SGX_port->writeRead("$PPG2," + QString::number(channel_number));

	if (checkStringContents(rx, "$PPG2,", "\r\n", ",ERR"))
	{
		QStringList get_list = rx.split(",");

		int count_chans = ((get_list.count() - 2) / 2);

		//
		// TODO:
		// It's wasteful that the check below is occurring every time power is fetched...
		//
		/* These Lists should be populated sufficiently to prevent crashes caused by index out of scope references, as well as to ensure the total power is accounted for properly in the GUI even if config is misconfigured!
		 * So use the argument count of PPG2 ensure that there are sufficient items in each list. */
		QList<QList<double>*> listlist {&PA_power_forward_watt_nchannel, &PA_power_reflected_watt_nchannel, &PA_s11_reflection_nchannel, &PA_power_forward_dbm_nchannel, &PA_power_reflected_dbm_nchannel, &PA_s11_nchannel};
		for(QList<double>* i: listlist)		//C++11 range-based for loop
		{
			while (i->count() < count_chans)
			{
				i->append(0.0);
			}
		}

		for (int i = 0; i < count_chans; i++)
		{
			PA_power_forward_watt_nchannel[i] = QString(get_list.at(2+i*2)).toDouble();
			PA_power_reflected_watt_nchannel[i] = QString(get_list.at(3+i*2)).toDouble();

			/* Compensate power readings for the PWM Duty Cycle */
			if (config->get_duty_cycle_compensation_enabled())
			{
				PA_power_forward_watt_nchannel[i] *= (PWM_duty_cycle * 0.01);
				PA_power_reflected_watt_nchannel[i] *= (PWM_duty_cycle * 0.01);
			}

			PA_power_forward_dbm_nchannel[i] = convert_watt_to_dbm(PA_power_forward_watt_nchannel[i]);
			PA_power_reflected_dbm_nchannel[i] = convert_watt_to_dbm(PA_power_reflected_watt_nchannel[i]);
		}

		Read_PA_Powers_interpret_readings_NChannel();
	}
}

/* Alternative PA power Get for a single SGx device controlling multiple PAs.
 * Get the individual FWD + RFL powers of each PA (in Watt) */
void RF_Channel::Read_PA_Powers_dBm_NChannel()
{
	if (!channel_working)
	{
		return;
	}

	QString rx = "";
	rx = SGX_port->writeRead("$PPDG2," + QString::number(channel_number));

	if (checkStringContents(rx, "$PPDG2,", "\r\n", ",ERR"))
	{
		QStringList get_list = rx.split(",");

		int count_chans = ((get_list.count() - 2) / 2);

		//
		// TODO:
		// It's wasteful that the check below is occurring every time power is fetched...
		//
		/* These Lists should be populated sufficiently to prevent crashes caused by index out of scope references, as well as to ensure the total power is accounted for properly in the GUI even if config is misconfigured!
		 * So use the argument count of PPDG2 ensure that there are sufficient items in each list. */
		QList<QList<double>*> listlist {&PA_power_forward_watt_nchannel, &PA_power_reflected_watt_nchannel, &PA_s11_reflection_nchannel, &PA_power_forward_dbm_nchannel, &PA_power_reflected_dbm_nchannel, &PA_s11_nchannel};
		for(QList<double>* i: listlist)		//C++11 range-based for loop
		{
			while (i->count() < count_chans)
			{
				i->append(0.0);
			}
		}

		for (int i = 0; i < count_chans; i++)
		{
			PA_power_forward_dbm_nchannel[i] = QString(get_list.at(2+i*2)).toDouble();
			PA_power_reflected_dbm_nchannel[i] = QString(get_list.at(3+i*2)).toDouble();

			/* Compensate power readings for the PWM Duty Cycle */
			if (config->get_duty_cycle_compensation_enabled())
			{
				PA_power_forward_dbm_nchannel[i] += convert_percent_to_dB(PWM_duty_cycle);
				PA_power_reflected_dbm_nchannel[i] += convert_percent_to_dB(PWM_duty_cycle);
			}

			PA_power_forward_watt_nchannel[i] = convert_dbm_to_watt(PA_power_forward_dbm_nchannel[i]);
			PA_power_reflected_watt_nchannel[i] = convert_dbm_to_watt(PA_power_reflected_dbm_nchannel[i]);
		}

		Read_PA_Powers_interpret_readings_NChannel();
	}
}

void RF_Channel::Read_PA_Powers_interpret_readings_NChannel()
{
	QList<double> FWD_dbm, RFL_dbm, S11, FWD_watt, RFL_watt, S11_reflection;

	Read_PA_Powers_calc_S11_dBm_NChannel(FWD_dbm, RFL_dbm, S11);
	Read_PA_Powers_calc_S11_watt_NChannel(FWD_watt, RFL_watt, S11_reflection);

	for (int i = 0; i < splitter_channel_count; i++)
	{
		emit signal_PA_power_readings_NChannel(intra_system_number, i+1, FWD_dbm[i], RFL_dbm[i], S11[i], FWD_watt[i], RFL_watt[i], S11_reflection[i]);
	}
}

/* Convert power and S11 values to logarithmic format */
void RF_Channel::Read_PA_Powers_calc_S11_dBm_NChannel(QList<double> &FWD, QList<double> &RFL, QList<double> &S11)
{
	for (int i = 0; i < splitter_channel_count; i++)
	{
		PA_s11_nchannel[i] = PA_power_reflected_dbm_nchannel[i] - PA_power_forward_dbm_nchannel[i];

		FWD.append(PA_power_forward_dbm_nchannel[i]);
		RFL.append(PA_power_reflected_dbm_nchannel[i]);
		S11.append(PA_s11_nchannel[i]);
	}
}

/* Convert power and S11 values to linear format */
void RF_Channel::Read_PA_Powers_calc_S11_watt_NChannel(QList<double> &FWD, QList<double> &RFL, QList<double> &S11)
{
	for (int i = 0; i < splitter_channel_count; i++)
	{
		PA_s11_nchannel[i] = (PA_power_reflected_watt_nchannel[i] / PA_power_forward_dbm_nchannel[i]) * 100;

		FWD.append(PA_power_forward_watt_nchannel[i]);
		RFL.append(PA_power_reflected_watt_nchannel[i]);
		S11.append(PA_s11_reflection_nchannel[i]);
	}
}

/**********************************************************************************************************************************************************************************
 * Temperature: get the temperature of the PA in degrees celsius
 *********************************************************************************************************************************************************************************/
void RF_Channel::Get_PA_Temperature()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PTG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PTG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		temperature_PA = QString(get_list.at(2)).toDouble();
	}

	//Calculate Average Temperature; temperature_Array serves as FIFO buffer for the values.
	temperature_PA_average = 0;
	if (temperature_PA_array.count() >= config->get_read_temperature())
	{
		temperature_PA_array.removeFirst();
	}

	temperature_PA_array.append(temperature_PA);

	for (int i = 0; i < temperature_PA_array.count(); i++)
	{
		temperature_PA_average += temperature_PA_array.at(i);
	}

	temperature_PA_average /= temperature_PA_array.count();
}

/**********************************************************************************************************************************************************************************
 * Temperature: get the temperature of the SG board in degrees celsius
 *********************************************************************************************************************************************************************************/
void RF_Channel::Get_SG_Temperature()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$TCG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$TCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		temperature_SG = QString(get_list.at(2)).toDouble();
	}

	//Calculate Average Temperature; temperature_Array serves as FIFO buffer for the values.
	temperature_SG_average = 0;
	if (temperature_SG_array.count() >= config->get_read_temperature_SG())
	{
		temperature_SG_array.removeFirst();
	}

	temperature_SG_array.append(temperature_SG);

	for (int i = 0; i < temperature_SG_array.count(); i++)
	{
		temperature_SG_average += temperature_SG_array.at(i);
	}

	temperature_SG_average /= temperature_SG_array.count();
}

/**********************************************************************************************************************************************************************************
 * RF enable: Set and Get the RF enable state.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_RF_enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "set_RF_enable(" << intrasys_num << "):" << enable;
	QString rx = SGX_port->writeRead("$ECS," + QString::number(channel_number) + "," + QString::number((int)enable));

	if (checkStringContents(rx, "$ECS,", ",OK\r\n", ",ERR"))
	{
		RF_enable = enable;
		emit signal_RF_enable_get(intra_system_number, RF_enable);	//emit that the update has occurred

#if defined (Q_OS_LINUX)
		if (config->get_support_HAT_B14_0835())
		{
			determine_LED_mode();
		}
#endif
	}
}

void RF_Channel::Get_RF_enable()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$ECG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$ECG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		RF_enable = (bool)QString(get_list.at(2)).toInt();
	}
}

/**********************************************************************************************************************************************************************************
 * Firmware Version: Get the firmware version.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Get_Firmware_Version()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$VER," + QString::number(channel_number));

	if (checkStringContents(get_string, "$VER,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");

		for (int i = 0; i < 4; i++)
		{
			firmware_version[i] = 0;
		}

		firmware_version[0] = QString(get_list.at(3)).toInt();
		firmware_version[1] = QString(get_list.at(4)).toInt();
		firmware_version[2] = QString(get_list.at(5)).toInt();

		QRegExp numbercheck("\\d+");  // a digit (\d), zero or more times (*)
		if (numbercheck.exactMatch(QString(get_list.at(6))))
		{
			firmware_version[3] = QString(get_list.at(6)).toInt();
		}

		firmware_version_string = QString::number(firmware_version[0]) + "." + QString::number(firmware_version[1]) + "." +QString::number(firmware_version[2]);
		if (firmware_version[3] > 0)
		{
			firmware_version_string += ("." + QString::number(firmware_version[3]));
		}
	}
}

/**********************************************************************************************************************************************************************************
 * Power: set / get the RF power value
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_Power_dBm(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	/* check if software needs to sanitize inputs */
	if (config->get_enforce_software_limits_power())
	{
		qDebug() << "power setpoint must be between" << convert_watt_to_dbm(config->get_power_watt_min()) << "and" << convert_watt_to_dbm(config->get_power_watt_max()) << "dBm.";

		/* Either auto-correct the input to the defined limits or deny the input altogether */
		if (config->get_enforce_software_limits_autocorrect())
		{
			if (val > convert_watt_to_dbm(config->get_power_watt_max())){
				val = convert_watt_to_dbm(config->get_power_watt_max());
			}
			else if (val < convert_watt_to_dbm(config->get_power_watt_min())){
				val = convert_watt_to_dbm(config->get_power_watt_min());
			}
		}
		else if (!(val <= convert_watt_to_dbm(config->get_power_watt_max()) && val >= convert_watt_to_dbm(config->get_power_watt_min())))
		{
			emit signal_power_get(intra_system_number, power_dbm, power_watt);
			return;
		}
	}

	//
	// TODO:
	// Hard-coded limit to prevent software from breaking.
	// if power is less than 0 dBm --> Numpad gets stuck at negative dBm input.
	// if power is less than -10 dBm --> Power in watt becomes 0W, which is not supported
	//
	if (val < 0)
	{
		val = 0;
	}

	qDebug() << "Set_Power_dbm -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$PWRDS," + QString::number(channel_number) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$PWRDS,", ",OK\r\n", ",ERR"))
	{
		power_dbm = val;
		power_watt = convert_dbm_to_watt(val);
		emit signal_power_get(intra_system_number, power_dbm, power_watt);	//emit that the update has occurred
	}
}

void RF_Channel::Set_Power_Watt(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	/* check if software needs to sanitize inputs */
	if (config->get_enforce_software_limits_power())
	{
		qDebug() << "Power setpoint must be between" << config->get_power_watt_min() << "and" << config->get_power_watt_max() << "watt.";

		/* Either auto-correct the input to the defined limits or deny the input altogether */
		if (config->get_enforce_software_limits_autocorrect())
		{
			if (val > config->get_power_watt_max()){
				val = config->get_power_watt_max();
			}
			else if (val < config->get_power_watt_min()){
				val = config->get_power_watt_min();
			}
		}
		else if (!(val <= config->get_power_watt_max() && val >= config->get_power_watt_min()))
		{
			emit signal_power_get(intra_system_number, power_dbm, power_watt);
			return;
		}
	}

	//
	// TODO:
	// Hard-coded limit to prevent software from breaking.
	// if power is less than 0.001 watt --> Numpad gets stuck at negative dBm input.
	// if power is 0W --> power in dBm becomes infinitely small
	//
	if (val < 0.001)
	{
		val = 0.001;
	}

	qDebug() << "Set_Power_Watt -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$PWRS," + QString::number(channel_number) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$PWRS,", ",OK\r\n", ",ERR"))
	{
		power_watt = val;
		power_dbm = convert_watt_to_dbm(val);
		emit signal_power_get(intra_system_number, power_dbm, power_watt);
	}
	else
	{
		qDebug() << "No OK!\t" << rx;
	}
}

void RF_Channel::Get_Power_dBm()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PWRDG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PWRDG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		power_dbm = QString(get_list.at(2)).toDouble();
		power_watt = convert_dbm_to_watt(power_dbm);
	}
}

void RF_Channel::Get_Power_watt()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PWRG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PWRG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		power_watt = QString(get_list.at(2)).toDouble();
		power_dbm = convert_watt_to_dbm(power_watt);
	}
}

/**********************************************************************************************************************************************************************************
 * Feed Forward Power: set / get the RF power during feed-forward control
 *********************************************************************************************************************************************************************************/

/* Set SGx board output power */
void RF_Channel::Set_SGxPower_dBm(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_SGxPower_dbm -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$PWRSGDS," + QString::number(channel_number) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$PWRSGDS,", ",OK\r\n", ",ERR"))
	{
		SGx_power_dBm = val;
		emit signal_sgx_power_dbm_get(intra_system_number, SGx_power_dBm);
	}
	else
	{
		qDebug() << "No OK!\t" << rx;
	}
}

/* Set IQ modulator Magnitude */
void RF_Channel::Set_IQMod_Magnitude(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	int temp_chan = channel_number;
	if (use_substitute_channel_number == true)
	{
		temp_chan = substitute_channel_number;
	}

	qDebug() << "Set_Magnitude -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$MCS," + QString::number(temp_chan) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$MCS,", ",OK\r\n", ",ERR"))
	{
		IQMod_magnitude = val;
		emit signal_IQMod_magnitude_get(intra_system_number, IQMod_magnitude);
	}
	else
	{
		qDebug() << "No OK!\t" << rx;
	}
}

/* Get IQ modulator Magnitude */
void RF_Channel::Get_IQMod_Magnitude()
{
	if (!channel_working)
	{
		return;
	}

	int temp_chan = channel_number;
	if (use_substitute_channel_number == true)
	{
		temp_chan = substitute_channel_number;
	}

	QString get_string = SGX_port->writeRead("$MCG," + QString::number(temp_chan));

	if (checkStringContents(get_string, "$MCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		IQMod_magnitude = QString(get_list.at(2)).toDouble();
	}
}

/* Set the VGA attentuation */
void RF_Channel::Set_VGA_attenuation(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	int temp_chan = channel_number;
	if (use_substitute_channel_number == true)
	{
		temp_chan = substitute_channel_number;
	}

	qDebug() << "Set_VGA_attenuation -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$GCS," + QString::number(temp_chan) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$GCS,", ",OK\r\n", ",ERR"))
	{
		VGA_attenuation = val;
		emit signal_VGA_attenuation_get(intra_system_number, VGA_attenuation);
	}
}

/* Get the VGA attentuation value */
void RF_Channel::Get_VGA_attenuation()
{
	if (!channel_working)
	{
		return;
	}

	int temp_chan = channel_number;
	if (use_substitute_channel_number == true)
	{
		temp_chan = substitute_channel_number;
	}

	QString get_string = SGX_port->writeRead("$GCG," + QString::number(temp_chan));

	if (checkStringContents(get_string, "$GCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		VGA_attenuation = QString(get_list.at(2)).toDouble();
	}
}


/**********************************************************************************************************************************************************************************
 * Frequency: set / get the basic RF frequency value
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_Frequency(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_Frequency -> input value as string: " << QString::number(val,'f',5);

	/* check if software needs to sanitize inputs */
	if (config->get_enforce_software_limits_frequency() == true)
	{
		qDebug() << "Frequency setpoint must be between" << config->get_frequency_min() << "and" << config->get_frequency_max() << "MHz.";

		/* Either auto-correct the input to the defined limits or deny the input altogether */
		if (config->get_enforce_software_limits_autocorrect() == true)
		{
			if (val > config->get_frequency_max()){
				val = config->get_frequency_max();
			}
			else if (val < config->get_frequency_min())
			{
				val = config->get_frequency_min();
			}
		}
		else if (!(val <= config->get_frequency_max() && val >= config->get_frequency_min()))
		{
			emit signal_frequency_get(intra_system_number, frequency_mhz);
			return;
		}
	}

	QString rx = SGX_port->writeRead("$FCS," + QString::number(channel_number) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$FCS,", ",OK\r\n", ",ERR"))
	{
		frequency_mhz = val;
		emit signal_frequency_get(intra_system_number, frequency_mhz);	//emit that the update has occurred
	}
}

void RF_Channel::Get_Frequency()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$FCG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$FCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		frequency_mhz = QString(get_list.at(2)).toDouble();
	}
}

/**********************************************************************************************************************************************************************************
 * Offset: set / get offset value for the RF power in dB
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_Offset_dB(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_Offset_dB -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$PODS," + QString::number(channel_number) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$PODS,", ",OK\r\n", ",ERR"))
	{
		offset_db = val;
		emit signal_offset_db_get(intra_system_number, offset_db);	//emit that the update has occurred
	}
}

void RF_Channel::Get_Offset_dB()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PODG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PODG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		offset_db = get_list.at(2).toDouble();
	}
}


/**********************************************************************************************************************************************************************************
 * Phase: set / get all manner of phase values of the RF signal in degrees
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_Phase(int intrasys_num, double val)
{
	if (!channel_working || config->get_support_phase_control() == false || intrasys_num != intra_system_number)
	{
		return;
	}

	if (val < 0)
	{
		val = 360 + val;	//make sure phase is always a positive number
	}

	int temp_chan = channel_number;
	if (use_substitute_channel_number == true)
	{
		temp_chan = substitute_channel_number;
	}

	qDebug() << "Set_Phase -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$PCS," + QString::number(temp_chan) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$PCS,", ",OK\r\n", ",ERR"))
	{
		phase_degrees = val;
		emit signal_phase_get(intra_system_number, phase_degrees);	//emit that the update has occurred
	}
}

void RF_Channel::Get_Phase()
{
	if (!channel_working || config->get_support_phase_control() == false)
	{
		return;
	}

	int temp_chan = channel_number;
	if (use_substitute_channel_number == true)
	{
		temp_chan = substitute_channel_number;
	}

	QString get_string = SGX_port->writeRead("$PCG," + QString::number(temp_chan));

	if (checkStringContents(get_string, "$PCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");

		double temp_phase = QString(get_list.at(2)).toDouble();
		if (temp_phase < 0)
		{
			temp_phase = 360 + temp_phase;	//make sure phase is always a positive number
		}
		phase_degrees = temp_phase;
	}
}

//Autophase functions.
void RF_Channel::Set_AutoPhase(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (val < 0)
	{
		val = 360 + val;	//make sure phase is always a positive number
	}
	qDebug() << "Set_AutoPhase -> input value as string: " << QString::number(val,'f',2);	//Command only supports precision of 2, more digits are fine but get ignored.
	QString rx = SGX_port->writeRead("$APTS," + QString::number(channel_number) + "," + QString::number(val,'f',2));

	if (checkStringContents(rx, "$APTS,", ",OK\r\n", ",ERR"))
	{
		autophase_degrees = val;
		emit signal_autophase_get(intra_system_number, autophase_degrees);	//emit that the update has occurred
	}
}

void RF_Channel::Get_AutoPhase()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$APTG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$APTG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");

		double temp_phase = QString(get_list.at(2)).toDouble();
		if (temp_phase < 0)
		{
			temp_phase = 360 + temp_phase;	//make sure phase is always a positive number
		}
		autophase_degrees = temp_phase;
	}
}


void RF_Channel::Set_AutoPhase_Enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_AutoPhase_Enable -> input value as string: " << QString::number(enable);
	QString rx = SGX_port->writeRead("$APES," + QString::number(channel_number) + "," + QString::number(enable));

	if (checkStringContents(rx, "$APES,", ",OK\r\n", ",ERR"))
	{
		autophase_enable = enable;
		emit signal_autophase_enable_get(intra_system_number, autophase_enable);	//emit that the update has occurred
	}
}

void RF_Channel::Get_AutoPhase_Enable()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$APEG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$APEG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		autophase_enable = (bool)QString(get_list.at(2)).toInt();
	}
}

void RF_Channel::Get_Phase_Demodulator()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$SPHASE," + QString::number(channel_number) + ",1");	//Get the Forward phase...

	if (checkStringContents(get_string, "$SPHASE,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");

		double temp_phase = QString(get_list.at(2)).toDouble();
		if (temp_phase < 0)
		{
			temp_phase = 360 + temp_phase;	//make sure phase is always a positive number
		}
		phase_degrees_demodulator = temp_phase;
	}
}


/**********************************************************************************************************************************************************************************
 * Splitter Board / NChannel: set / get phases of different subchannels a Splitter Board
 *********************************************************************************************************************************************************************************/
/* Set the Phase Gain Board phase value */
void RF_Channel::Set_Phase_NChannel(int intrasys_num, int subchannel_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (val < 0)
	{
		val = 360 + val;	//make sure phase is always a positive number
	}

	qDebug() << "Set_Phase_NChannel(" + QString::number(subchannel_num) + ") -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$PCS," + QString::number(subchannel_num) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$PCS,", ",OK\r\n", ",ERR"))
	{
		phase_degrees_nchannel[subchannel_num - 1] = val;
	}

	emit signal_NChannel_phase_get(intra_system_number, subchannel_num, phase_degrees_nchannel[subchannel_num-1]);		//emit that the update has occurred
}

/* Get the Phase Gain Board phase value */
void RF_Channel::Get_Phase_NChannel(int subchannel_num)
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PCG," + QString::number(subchannel_num));

	if (checkStringContents(get_string, "$PCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");

		double temp_phase = QString(get_list.at(2)).toDouble();
		if (temp_phase < 0)
		{
			temp_phase = 360 + temp_phase;	//make sure phase is always a positive number
		}
		phase_degrees_nchannel[subchannel_num-1] = temp_phase;
	}
}

void RF_Channel::Set_Attenuation_NChannel(int intrasys_num, int subchannel_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_Attenuation_NChannel(" + QString::number(subchannel_num) + ") -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$GCS," + QString::number(subchannel_num) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$GCS,", ",OK\r\n", ",ERR"))
	{
		attenuation_nchannel[subchannel_num - 1] = val;
	}

	emit signal_NChannel_attenuation_get(intra_system_number, subchannel_num, attenuation_nchannel[subchannel_num-1]);		//emit that the update has occurred
}

void RF_Channel::Get_Attenuation_NChannel(int subchannel_num)
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$GCG," + QString::number(subchannel_num));

	if (checkStringContents(get_string, "$GCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		attenuation_nchannel[subchannel_num-1] = QString(get_list.at(2)).toDouble();
	}
}

void RF_Channel::Set_Magnitude_NChannel(int intrasys_num, int subchannel_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_Magnitude_NChannel(" + QString::number(subchannel_num) + ") -> input value as string: " << QString::number(val,'f',5);
	QString rx = SGX_port->writeRead("$MCS," + QString::number(subchannel_num) + "," + QString::number(val,'f',5));

	if (checkStringContents(rx, "$MCS,", ",OK\r\n", ",ERR"))
	{
		magnitude_nchannel[subchannel_num - 1] = val;
	}

	emit signal_NChannel_magnitude_get(intra_system_number, subchannel_num, magnitude_nchannel[subchannel_num-1]);		//emit that the update has occurred
}

void RF_Channel::Get_Magnitude_NChannel(int subchannel_num)
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$MCG," + QString::number(subchannel_num));

	if (checkStringContents(get_string, "$MCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		magnitude_nchannel[subchannel_num-1] = QString(get_list.at(2)).toDouble();
	}
}

/**********************************************************************************************************************************************************************************
 * Demodulator: set / get the enable state the IQ Demod board.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_Demodulator_Enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_Demodulator_Enable -> input value as string: " << QString::number(enable);
	QString rx = SGX_port->writeRead("$DES," + QString::number(channel_number) + "," + QString::number(enable));

	if (checkStringContents(rx, "$DES,", ",OK\r\n", ",ERR"))
	{
		demodulator_enable = enable;
		emit signal_demodulator_enable_get(intra_system_number, demodulator_enable);	//emit that the update has occurred
	}
}

void RF_Channel::Get_Demodulator_Enable()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$DEG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$DEG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		demodulator_enable = (bool)QString(get_list.at(2)).toInt();
	}
}

/**********************************************************************************************************************************************************************************
 * Clock Source: set / get the clock source mode
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_Clock_Source(int intrasys_num, int mode)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_Clock_Source -> input value as string: " << QString::number(mode);
	QString rx = SGX_port->writeRead("$CSS," + QString::number(channel_number) + "," + QString::number(mode));

	if (checkStringContents(rx, "$CSS,", ",OK\r\n", ",ERR"))
	{
		clock_source = mode;
		emit signal_clock_source_get(intra_system_number, clock_source);	//emit that the update has occurred
	}
}

void RF_Channel::Get_Clock_Source()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$CSG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$CSG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		clock_source = QString(get_list.at(2)).toInt();
	}
}

/**********************************************************************************************************************************************************************************
 * PWM Triggering: set / get the PWM triggering mode
 *
 * Currently this is not supported! Any input other than mode 1 causes a crash to occur on the SGx board.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_PWM_Triggering(int intrasys_num, int mode)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "This functionality is not yet supported.";
	qDebug() << "Set_PWM_Triggering -> input value as string: " << QString::number(1);
	QString rx = SGX_port->writeRead("$DCTS," + QString::number(channel_number) + "," + QString::number(1));

	if (checkStringContents(rx, "$DCTS,", ",OK\r\n", ",ERR"))
	{
		PWM_triggering_mode = mode;
		emit signal_PWM_triggering_get(intra_system_number, PWM_triggering_mode);	//emit that the update has occurred
	}

//	qDebug() << "Set_PWM_Triggering -> input value as string: " << QString::number(mode);
//	QString rx = SGX_port->writeRead("$DCTS"
//									 + "," + QString::number(channel_number)
//									 + "," + QString::number(mode)
//									 + "," + QString::number(config->get_PWM_slave_port())
//									 + "," + QString::number(config->get_PWM_slave_pin())
//									 + "," + QString::number(config->get_PWM_master_port())
//									 + "," + QString::number(config->get_PWM_master_pin()));

//	if (checkStringContents(rx, "$DCTS,", ",OK\r\n", ",ERR"))
//	{
//		PWM_triggering_mode = mode;
//		PWM_triggering_slave_port = config->get_PWM_slave_port();
//		PWM_triggering_slave_pin = config->get_PWM_slave_pin();
//		PWM_triggering_master_port = config->get_PWM_master_port();
//		PWM_triggering_master_pin = config->get_PWM_master_pin();
//		emit signal_PWM_triggering_get(intra_system_number, PWM_triggering_mode);	//emit that the update has occurred
//	}

}

//The Get is function $DCG, the universal get for everything PWM.


/**********************************************************************************************************************************************************************************
 * Analog Input: set / get the analog input mode
 *********************************************************************************************************************************************************************************/
//
// TODO:
// Analog Input Mode lacks a GET function
// Blocked by: Missing $command in firmware.
//

/* Not a slot */
void RF_Channel::Set_Analog_Input_Enable(bool enable)
{
	if (!channel_working)
	{
		return;
	}

	analog_input_enable = enable;
	Set_Analog_Input_Settings();
}

/* Not a slot */
void RF_Channel::Set_Analog_Input_Modulator(int modulator)
{
	if (!channel_working)
	{
		return;
	}

	analog_input_modulator = modulator;
	Set_Analog_Input_Settings();
}

/* Not a slot */
void RF_Channel::Set_Analog_Input_Settings()
{
	if (!channel_working)
	{
		return;
	}

	QString rx = "";

	if (analog_input_modulator == 0)		//Modulation source = IQ mod
	{
		rx = SGX_port->writeRead("$AIS," + QString::number(channel_number)
								 + "," + QString::number((int)analog_input_enable)
								 + "," + QString::number(analog_input_modulator)
								 + "," + QString::number(config->get_AIS_magnitude_min(),'f',5)
								 + "," + QString::number(config->get_AIS_ADC_min())
								 + "," + QString::number(config->get_AIS_magnitude_max(),'f',5)
								 + "," + QString::number(config->get_AIS_ADC_max()));

		if (analog_input_enable == true)
		{
			/* This sets the default Attenuation high as a safety precaution, however the user can still change the value from the Home page of the GUI */
			Set_VGA_attenuation(intra_system_number, analog_input_VGA_attenuation);

			//
			// TODO:
			// This part is kind of jank.
			// It can make for some awkward interactions when switching between Analog and FeedFWD mode.
			// Ideally I wouldn't want to be doing this at all, and instead rely on something like $FFGCS. But that only seems to work if there's a FF cal present(?)
			//
		}
	}
	else if (analog_input_modulator == 1)	//Modulation source = VGA
	{
		rx = SGX_port->writeRead("$AIS," + QString::number(channel_number)
								 + "," + QString::number((int)analog_input_enable)
								 + "," + QString::number(analog_input_modulator)
								 + "," + QString::number(config->get_AIS_attenuation_max_db(),'f',5)
								 + "," + QString::number(config->get_AIS_ADC_min())
								 + "," + QString::number(config->get_AIS_attenuation_min_db(),'f',5)
								 + "," + QString::number(config->get_AIS_ADC_max()));
	}

	//
	// TODO:
	// Something should be happening with $MCS here.
	//

	/* Remark:
	 * Autogain is automatically disabled when analog input mode is enabled with $AIS.
	 * Autogain must be manually enabled after analog input mode is turned off with $AIS.
	 * This is covered by the Set_Power_Control_Mode() function. */

	if (checkStringContents(rx, "$AIS,", ",OK\r\n", ",ERR"))
	{
		//
		// TODO:
		// Firmware is missing a $AIG needed for Power Control Mode polling...
		//
	}

	qDebug() << "Set_Analog_Input_Settings -> input value as string: " << QString::number(analog_input_enable) << QString::number(analog_input_modulator);
}

/**********************************************************************************************************************************************************************************
 * External triggering: set / get the analog input mode
 *********************************************************************************************************************************************************************************/
/* Overarching slot function for external triggering that will manage all the sets in one go */
void RF_Channel::Set_External_Triggering_Enable_Configuration(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	//
	// TODO:
	// External Triggering (without synchronization) sabotages power measurements and should therefore be incompatible with auto-gain.
	// Enabling external triggering should make auto-gain power control mode become unavailable in the settings menu.
	// External triggering should even disable the possibility to press the CW button, as it would change the PWM duty cycle from 0 to 100%
	//

	if (config->get_external_triggering_synchronization_enable() > -1)
	{
		Set_External_Triggering_Synchronization_Enable(intra_system_number, (enable && config->get_external_triggering_synchronization_enable()));
	}

	if (config->get_external_triggering_synchronization_delay_us() > -1)
	{
		Set_External_Triggering_Synchronization_Delay(intra_system_number, (int)enable * external_triggering_synchronization_delay_us_target);
	}

	Set_External_Triggering_Enable(intra_system_number, enable);

	if (external_triggering_enable == enable)
	{
		if (enable == true)
		{
			Set_PWM_enable(intra_system_number, false);		//PWM is never compatible with external triggering

			if (external_triggering_synchronization_enable < 1)
			{
				Set_ALL_enable(intra_system_number, false);
				Set_DLL_enable(intra_system_number, false);
			}
		}

		emit signal_external_triggering_enable_get(intra_system_number, external_triggering_enable);	//emit that the update has occurred
		emit signal_CW_enable_get(intra_system_number, !(ALL_enable || DLL_enable || PWM_enable));
		determine_availability_PWM();
		determine_availability_amplifier_mode();
		determine_availability_sweep();
		determine_availability_DLL();
		determine_availability_ALL();
	}
}

void RF_Channel::Set_External_Triggering_Enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_External_Triggering_Enable(" << intrasys_num << "):" << enable;
	QString rx = SGX_port->writeRead("$ETS," + QString::number(channel_number) + "," + QString::number((int)enable));

	if (checkStringContents(rx, "$ETS,", ",OK\r\n", ",ERR"))
	{
		external_triggering_enable = enable;
		emit signal_external_triggering_enable_get(intra_system_number, external_triggering_enable);	//emit that the update has occurred

		/* Remark: $ETS auto-toggles autogain */
		autogain_enable = !enable;
		poll_power_control_mode();		//This sends $RFSG and $AGEG, which is a bit excessive tbh.
	}
}

void RF_Channel::Get_External_Triggering_Enable()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$ETG," + QString::number(channel_number));
	if (checkStringContents(get_string, "$ETG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		external_triggering_enable = (bool)QString(get_list.at(2)).toInt();
	}
}

void RF_Channel::Set_External_Triggering_Synchronization_Enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_External_Triggering_Synchronization_Enable(" << intrasys_num << "):" << enable;
	QString rx = SGX_port->writeRead("$ETSS," + QString::number(channel_number) + "," + QString::number((int)enable));

	if (checkStringContents(rx, "$ETSS,", ",OK\r\n", ",ERR"))
	{
		external_triggering_synchronization_enable = enable;
		emit signal_external_triggering_synchronization_enable_get(intra_system_number, external_triggering_synchronization_enable);
	}
}

void RF_Channel::Get_External_Triggering_Synchronization_Enable()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$ETSG," + QString::number(channel_number));
	if (checkStringContents(get_string, "$ETSG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		external_triggering_synchronization_enable = QString(get_list.at(2)).toInt();
	}
}

void RF_Channel::Set_External_Triggering_Synchronization_Delay(int intrasys_num, int useconds)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_External_Triggering_Synchronization_Delay(" << intrasys_num << "):" << useconds;
	QString rx = SGX_port->writeRead("$ETSDS," + QString::number(channel_number) + "," + QString::number(useconds));

	if (checkStringContents(rx, "$ETSDS,", ",OK\r\n", ",ERR"))
	{
		external_triggering_synchronization_delay_us = useconds;
		emit signal_external_triggering_synchronization_delay_get(intra_system_number, external_triggering_synchronization_delay_us);
	}
}

void RF_Channel::Get_External_Triggering_Synchronization_Delay()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$ETSDG," + QString::number(channel_number));
	if (checkStringContents(get_string, "$ETSDG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		external_triggering_synchronization_delay_us = QString(get_list.at(2)).toInt();
	}
}

/**********************************************************************************************************************************************************************************
 * Status : get the status of the SGx board to see if there are any error / Clear the errors with error clear.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Get_Status()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$ST," + QString::number(channel_number));

	if (checkStringContents(get_string, "$ST,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		error = QString(get_list.at(3)).toULongLong(nullptr,16);
	}
}

void RF_Channel::Execute_Error_Clear(int intrasys_num)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$ERRC," + QString::number(channel_number));

	if (checkStringContents(get_string, "$ERRC,", ",OK\r\n", ",ERR"))
	{
		//Nothing to do...
	}
}

void RF_Channel::Set_Error_Clearing_Enable(int intrasys_num, bool enable)
{
	//No channel_working check; For this function it doesn't really matter.
	if (intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Subsystem[" << intra_system_number << "]: Automatic error clearing" << ((enable == true) ? "enabled" : "disabled");
	error_clearing_enable = enable;
}


/**********************************************************************************************************************************************************************************
 * Runtime : Get the runtime / uptime of the SGx board.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Get_SGx_Runtime()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$RTG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$RTG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		runtime_SGx = QString(get_list.at(2)).toULongLong();
	}
}

/* Use the SGx board runtime / uptime to detect a reset.
 * Essentially a workaround for the reduction of situations that trigger an 0x20 reset error */
void RF_Channel::Detect_Runtime_SGx_Reset()
{
	quint64 last_runtime_SGx = Runtime_SGx();
	Get_SGx_Runtime();

	//If last_runtime > runtime --> Reset has occurred. Unless it's an overflow...
	if (last_runtime_SGx > Runtime_SGx())
	{
		qDebug() << "Subsystem[" << intra_system_number << "]: Runtime-based Reset Detected";
		emit signal_reset_detected_runtime(intra_system_number, channel_number);
	}
}


/**********************************************************************************************************************************************************************************
 * CW : set / get the enable state of CW mode
 *********************************************************************************************************************************************************************************/
//void RF_Channel::Set_CW_enable(int intrasys_num, bool enable)
//{
//	if (!channel_working || intrasys_num != intra_system_number)
//	{
//		return;
//	}

//	qDebug() << "set_DLL_enable(" << intrasys_num << "):" << enable;
//	QString rx = SGX_port->writeRead("$DLES," + QString::number(channel_number) + "," + QString::number((int)enable));
//	QString rx2 = SGX_port->writeRead("$ALES," + QString::number(channel_number) + "," + QString::number((int)false));

//	if (rx.contains(",OK\r\n") && rx2.contains(",OK\r\n"))
//	{
//		DLL_enable = enable;
//		ALL_enable = false;
//		emit signal_DLL_enable_get(intra_system_number, DLL_enable);	//emit that the update has occurred
//		emit signal_ALL_enable_get(intra_system_number, ALL_enable);	//emit that the update has occurred

//		if (ALL_enable == false && DLL_enable == false && PWM_enable == false)
//		{
//			emit signal_CW_enable_get(intra_system_number, true);
//		}
//	}
//}

/**********************************************************************************************************************************************************************************
 * DLL : set / get the enable state and configuration of DLL
 *********************************************************************************************************************************************************************************/

void RF_Channel::Set_DLL_enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "set_DLL_enable(" << intrasys_num << "):" << enable;
	QString rx = SGX_port->writeRead("$DLES," + QString::number(channel_number) + "," + QString::number((int)enable));

	if (ALL_compatible == true)
	{
		QString rx2 = SGX_port->writeRead("$ALES," + QString::number(channel_number) + "," + QString::number((int)false));

		if (checkStringContents(rx, "$DLES,", ",OK\r\n", ",ERR") &&
			checkStringContents(rx2, "$ALES,", ",OK\r\n", ",ERR"))
		{
			DLL_enable = enable;
			ALL_enable = false;
			emit signal_DLL_enable_get(intra_system_number, DLL_enable);	//emit that the update has occurred
			if (ALL_compatible == true)
			{
				emit signal_ALL_enable_get(intra_system_number, ALL_enable);	//emit that the update has occurred
			}
			Set_Analog_Input_Modulator(0);
		}
	}
	else
	{
		if (checkStringContents(rx, "$DLES,", ",OK\r\n", ",ERR"))
		{
			DLL_enable = enable;
			emit signal_DLL_enable_get(intra_system_number, DLL_enable);	//emit that the update has occurred
			Set_Analog_Input_Modulator(0);
		}
	}

	poll_CW_enable();
}

void RF_Channel::Get_DLL_enable()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$DLEG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$DLEG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		DLL_enable = QString(get_list.at(2)).toDouble();
	}
}

void RF_Channel::Set_DLL_Frequency_Lower_Limit(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (config->get_enforce_software_limits_frequency() == true)
	{
		/* Either auto-correct the input to the defined limits or deny the input altogether */
		if (config->get_enforce_software_limits_autocorrect() == true)
		{
			if (val < config->get_frequency_min()){
				val = config->get_frequency_min();
			}
			else if (val >= DLL_frequency_limit_upper){
				val = DLL_frequency_limit_upper-1;		//Don't set lower frequency higher than upper frequency
			}
		}
		else if (!(val <= (DLL_frequency_limit_upper-1) && val >= config->get_frequency_min()))
		{
			emit signal_DLL_settings_get(intra_system_number, DLL_frequency_limit_lower, DLL_frequency_limit_upper, DLL_start_frequency, DLL_step_frequency, DLL_threshold, DLL_main_delay);	//emit that the update has occurred
			return;
		}
	}

	DLL_frequency_limit_lower = val;
	Set_DLL_Settings();
}

void RF_Channel::Set_DLL_Frequency_Upper_Limit(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (config->get_enforce_software_limits_frequency() == true)
	{
		if (config->get_enforce_software_limits_autocorrect() == true)
		{
			if (val > config->get_frequency_max()){
				val = config->get_frequency_max();
			}
			else if (val <= DLL_frequency_limit_lower){
				val = DLL_frequency_limit_lower + 1;
			}
		}
		else if (!(val <= config->get_frequency_max() && val >= (DLL_frequency_limit_lower + 1)))
		{
			emit signal_DLL_settings_get(intra_system_number, DLL_frequency_limit_lower, DLL_frequency_limit_upper, DLL_start_frequency, DLL_step_frequency, DLL_threshold, DLL_main_delay);	//emit that the update has occurred
			return;
		}
	}

	DLL_frequency_limit_upper = val;
	Set_DLL_Settings();
}

void RF_Channel::Set_DLL_Frequency_Start(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (config->get_enforce_software_limits_frequency() == true)
	{
		if (config->get_enforce_software_limits_autocorrect() == true)
		{
			if (val >= DLL_frequency_limit_upper){
				val = DLL_frequency_limit_upper;
			}
			else if (val <= DLL_frequency_limit_lower){
				val = DLL_frequency_limit_lower;
			}
		}
		else if (!(val <= DLL_frequency_limit_upper && val >= DLL_frequency_limit_lower))
		{
			emit signal_DLL_settings_get(intra_system_number, DLL_frequency_limit_lower, DLL_frequency_limit_upper, DLL_start_frequency, DLL_step_frequency, DLL_threshold, DLL_main_delay);	//emit that the update has occurred
			return;
		}
	}

	DLL_start_frequency = val;
	Set_DLL_Settings();
}

void RF_Channel::Set_DLL_Frequency_Step(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (config->get_enforce_software_limits_frequency() == true)
	{
		if (config->get_enforce_software_limits_autocorrect() == true)
		{
			if (val > (DLL_frequency_limit_upper - DLL_frequency_limit_lower)){
				val = DLL_frequency_limit_upper - DLL_frequency_limit_lower;
			}
			else if (val < config->get_DLL_freqLimit_step()){
				val = config->get_DLL_freqLimit_step();
			}
		}
		else if (!(val >= config->get_DLL_freqLimit_step() && val <= (DLL_frequency_limit_upper - DLL_frequency_limit_lower)))
		{
			emit signal_DLL_settings_get(intra_system_number, DLL_frequency_limit_lower, DLL_frequency_limit_upper, DLL_start_frequency, DLL_step_frequency, DLL_threshold, DLL_main_delay);	//emit that the update has occurred
			return;
		}
	}

	DLL_step_frequency = val;
	Set_DLL_Settings();
}

void RF_Channel::Set_DLL_Frequency_Threshold(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	DLL_threshold = val;
	Set_DLL_Settings();
}

void RF_Channel::Set_DLL_Frequency_Main_Delay(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	DLL_main_delay = val;
	Set_DLL_Settings();
}

//Not a slot function
void RF_Channel::Set_DLL_Settings()
{
	if (!channel_working)
	{
		return;
	}

	/* (optionally) apply auto-corrections to the start and step frequencies when the upper / lower Frequency is changed */
	if (config->get_enforce_software_limits_frequency() == true && 	config->get_enforce_software_limits_autocorrect() == true)
	{
		Autocorrect_DLL_Minmax_Values();
	}

	qDebug() << "set_DLL_settings -> input value as string: " << "$DLCS," + QString::number(channel_number) + "," + QString::number(DLL_frequency_limit_lower,'f',5) + "," + QString::number(DLL_frequency_limit_upper,'f',5) + "," + QString::number(DLL_start_frequency,'f',5) + "," + QString::number(DLL_step_frequency,'f',5) + "," + QString::number(DLL_threshold,'f',5) + "," + QString::number(DLL_main_delay,'f',5);
	QString rx = SGX_port->writeRead("$DLCS,"
									 + QString::number(channel_number)
									 + "," + QString::number(DLL_frequency_limit_lower,'f',5)
									 + "," + QString::number(DLL_frequency_limit_upper,'f',5)
									 + "," + QString::number(DLL_start_frequency,'f',5)
									 + "," + QString::number(DLL_step_frequency,'f',5)
									 + "," + QString::number(DLL_threshold,'f',5)
									 + "," + QString::number(DLL_main_delay));

	if (!checkStringContents(rx, "$DLCS,", ",OK\r\n", ",ERR"))
	{
		Get_DLL_Settings();		//if setting the DLL value fails, set the variables back to normal.
	}

	emit signal_DLL_settings_get(intra_system_number, DLL_frequency_limit_lower, DLL_frequency_limit_upper, DLL_start_frequency, DLL_step_frequency, DLL_threshold, DLL_main_delay);	//emit that the update has occurred
}

void RF_Channel::Get_DLL_Settings()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$DLCG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$DLCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		DLL_frequency_limit_lower = QString(get_list.at(2)).toDouble();
		DLL_frequency_limit_upper = QString(get_list.at(3)).toDouble();
		DLL_start_frequency = QString(get_list.at(4)).toDouble();
		DLL_step_frequency = QString(get_list.at(5)).toDouble();
		DLL_threshold = QString(get_list.at(6)).toDouble();
		DLL_main_delay = QString(get_list.at(7)).toDouble();
	}
}

/* Auto-correct the DLL frequency values that are outside the defined frequency limits */
void RF_Channel::Autocorrect_DLL_Minmax_Values()
{
	if (DLL_start_frequency > DLL_frequency_limit_upper)
	{
		DLL_start_frequency = DLL_frequency_limit_upper;
	}
	else if (DLL_start_frequency < DLL_frequency_limit_lower)
	{
		DLL_start_frequency = DLL_frequency_limit_lower;
	}

	if (DLL_step_frequency > (DLL_frequency_limit_upper - DLL_frequency_limit_lower))
	{
		DLL_step_frequency = (DLL_frequency_limit_upper - DLL_frequency_limit_lower);
	}
	else if (DLL_step_frequency < config->get_DLL_freqLimit_step())
	{
		DLL_step_frequency = config->get_DLL_freqLimit_step();
	}
}

/**********************************************************************************************************************************************************************************
 * ALL : set / get the enable state and configuration of ALL
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_ALL_enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number || !ALL_compatible)
	{
		return;
	}

	qDebug() << "set_ALL_enable(" << intrasys_num << "):" << enable;
	QString rx = SGX_port->writeRead("$ALES," + QString::number(channel_number) + "," + QString::number((int)enable));
	QString rx2 = SGX_port->writeRead("$DLES," + QString::number(channel_number) + "," + QString::number(false));	//Disable ALL.

	if (checkStringContents(rx, "$ALES,", ",OK\r\n", ",ERR") &&
		checkStringContents(rx2, "$DLES,", ",OK\r\n", ",ERR"))
	{
		ALL_enable = enable;
		DLL_enable = false;
		emit signal_ALL_enable_get(intra_system_number, ALL_enable);	//emit that the update has occurred
		emit signal_DLL_enable_get(intra_system_number, DLL_enable);	//emit that the update has occurred

		poll_CW_enable();
	}

	if (ALL_enable == true)
	{
		Set_Analog_Input_Modulator(1);
	}
	else
	{
		Set_Analog_Input_Modulator(0);
	}
}

void RF_Channel::Get_ALL_enable()
{
	if (!channel_working || !ALL_compatible)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$ALEG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$ALEG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		ALL_enable = (bool)QString(get_list.at(2)).toInt();

		if (ALL_enable == true)
		{
			if (analog_input_modulator != 1)
			{
				Set_Analog_Input_Modulator(1);
			}
		}
		else
		{
			if (analog_input_modulator != 0)
			{
				Set_Analog_Input_Modulator(0);
			}
		}
	}
}

void RF_Channel::Set_ALL_Frequency_Lower_Limit(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	ALL_frequency_limit_lower = val;
	Set_ALL_Settings();
}

void RF_Channel::Set_ALL_Frequency_Upper_Limit(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	ALL_frequency_limit_upper = val;
	Set_ALL_Settings();
}

void RF_Channel::Set_ALL_Frequency_Threshold(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	ALL_threshold = val;
	Set_ALL_Settings();
}

//Not a slot function
void RF_Channel::Set_ALL_Settings()
{
	if (!channel_working || !ALL_compatible)
	{
		return;
	}

	qDebug() << "set_ALL_settings -> input value as string: " << "$ALCS," + QString::number(channel_number) + "," + QString::number(ALL_frequency_limit_lower,'f',5) + "," + QString::number(ALL_frequency_limit_upper,'f',5) + "," + QString::number(ALL_threshold,'f',5);
	QString rx = SGX_port->writeRead("$ALCS,"
									 + QString::number(channel_number)
									 + "," + QString::number(ALL_frequency_limit_lower,'f',5)
									 + "," + QString::number(ALL_frequency_limit_upper,'f',5)
									 + "," + QString::number(ALL_threshold,'f',5));

	if (checkStringContents(rx, "$ALCS,", ",OK\r\n", ",ERR"))
	{
		Get_ALL_Settings();		//if setting the ALL value fails, set the variables back to normal.
	}

	emit signal_ALL_settings_get(intra_system_number, ALL_frequency_limit_lower, ALL_frequency_limit_upper, ALL_threshold);	//emit that the update has occurred
}

void RF_Channel::Get_ALL_Settings()
{
	if (!channel_working || !ALL_compatible)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$ALCG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$ALCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		ALL_frequency_limit_lower = QString(get_list.at(2)).toDouble();
		ALL_frequency_limit_upper = QString(get_list.at(3)).toDouble();
		ALL_threshold = QString(get_list.at(4)).toDouble();
	}

	// Set's the target DVGA values for when ALL gets enabled.
	if (ALL_enable == true && DVGA_forward_ALL_enable_log == true)
	{
		target_DVGA_attenuation_forward = DVGA_attenuation_forward;
		target_DVGA_amplifier_forward_enable = DVGA_amplifier_forward_enable;
	}
	if (ALL_enable == true && DVGA_reflected_ALL_enable_log == true)
	{
		target_DVGA_attenuation_reflected = DVGA_attenuation_reflected;
		target_DVGA_amplifier_reflected_enable = DVGA_amplifier_reflected_enable;
	}
}

/**********************************************************************************************************************************************************************************
 * DVGA : set / get the state and configuration of DVGA's and amplifiers of the internal detectors.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_DVGA_Amplifier_Forward_Enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	DVGA_amplifier_forward_enable = enable;
	Set_DVGA_Forward_Settings();
}

void RF_Channel::Set_DVGA_Attenuation_Forward(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	DVGA_attenuation_forward = val;
	Set_DVGA_Forward_Settings();
}

//Not a slot
void RF_Channel::Set_DVGA_Forward_Settings()
{
	if (!channel_working || !DVGA_compatible)
	{
		return;
	}

	qDebug() << "Set_DVGA_Forward_Settings(" << intra_system_number << "):" << QString::number((int)DVGA_amplifier_forward_enable) << QString::number(DVGA_attenuation_forward,'f',5);
	QString rx = SGX_port->writeRead("$FWDS," + QString::number(channel_number) + "," + QString::number((int)DVGA_amplifier_forward_enable) + "," + QString::number(DVGA_attenuation_forward,'f',5));

	if (checkStringContents(rx, "$FWDS,", ",OK\r\n", ",ERR"))
	{
		Get_DVGA_Forward_Settings();		//if setting the values fails, set the variables back to normal.
	}

	emit signal_DVGA_forward_settings_get(intra_system_number, DVGA_amplifier_forward_enable, DVGA_attenuation_forward);	//emit that the update has occurred
}

//Not a slot
void RF_Channel::Get_DVGA_Forward_Settings()
{
	if (!channel_working || !DVGA_compatible)
	{
		return;
	}

	if (ALL_enable == false)
	{
		DVGA_forward_ALL_enable_log = ALL_enable;
	}

	QString get_string = SGX_port->writeRead("$FWDG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$FWDG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		DVGA_amplifier_forward_enable = (bool)QString(get_list.at(2)).toInt();
		DVGA_attenuation_forward = QString(get_list.at(3)).toDouble();
	}

	//
	// TODO:
	// When switching on ALL it's kind of stupid that Attenuation and Amplifer get written separately, rather than in one go.
	//

	//Set the DVGA's to the default state when ALL isn't enabled (For Macom PA compatibility)
	if (ALL_enable == false && (DVGA_attenuation_forward != config->get_default_DVGA_attenuation_forward() || DVGA_amplifier_forward_enable != config->get_default_DVGA_amplifier_forward_enabled()))
	{
		Set_DVGA_Attenuation_Forward(intra_system_number, config->get_default_DVGA_attenuation_forward());
		Set_DVGA_Amplifier_Forward_Enable(intra_system_number, config->get_default_DVGA_amplifier_forward_enabled());
	}

	if (ALL_enable == true && DVGA_forward_ALL_enable_log == false)	//went from ALL disabled to enabled
	{
		Set_DVGA_Attenuation_Forward(intra_system_number, target_DVGA_attenuation_forward);
		Set_DVGA_Amplifier_Forward_Enable(intra_system_number, target_DVGA_amplifier_forward_enable);
		DVGA_forward_ALL_enable_log = ALL_enable;
	}
}

void RF_Channel::Set_DVGA_Amplifier_Reflected_Enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	DVGA_amplifier_reflected_enable = enable;
	Set_DVGA_Reflected_Settings();
}

void RF_Channel::Set_DVGA_Attenuation_Reflected(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	DVGA_attenuation_reflected = val;
	Set_DVGA_Reflected_Settings();
}

//Not a slot
void RF_Channel::Set_DVGA_Reflected_Settings()
{
	if (!channel_working || !DVGA_compatible)
	{
		return;
	}

	qDebug() << "set_DVGA_Reflected_Settings(" << intra_system_number << "):" << QString::number((int)DVGA_amplifier_reflected_enable) << QString::number(DVGA_attenuation_reflected,'f',5);
	QString rx = SGX_port->writeRead("$RFLS," + QString::number(channel_number) + "," + QString::number((int)DVGA_amplifier_reflected_enable) + "," + QString::number(DVGA_attenuation_reflected,'f',5));

	if (checkStringContents(rx, "$RFLS,", ",OK\r\n", ",ERR"))
	{
		Get_DVGA_Reflected_Settings();		//if setting the values fails, set the variables back to normal.
	}

	emit signal_DVGA_reflected_settings_get(intra_system_number, DVGA_amplifier_reflected_enable, DVGA_attenuation_reflected);	//emit that the update has occurred
}

void RF_Channel::Get_DVGA_Reflected_Settings()
{
	if (!channel_working || !DVGA_compatible)
	{
		return;
	}

	if (ALL_enable == false)
	{
		DVGA_reflected_ALL_enable_log = ALL_enable;
	}

	QString get_string = SGX_port->writeRead("$RFLG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$RFLG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		DVGA_amplifier_reflected_enable = (bool)QString(get_list.at(2)).toInt();
		DVGA_attenuation_reflected = QString(get_list.at(3)).toDouble();
	}

	//
	// TODO:
	// When switching on ALL it's kind of stupid that Attenuation and Amplifer get written separately, rather than in one go.
	//

	//Set the DVGA's to the default state when ALL isn't enabled (For Macom PA compatibility)
	if (ALL_enable == false && (DVGA_attenuation_reflected != config->get_default_DVGA_attenuation_reflected() || DVGA_amplifier_reflected_enable != config->get_default_DVGA_amplifier_reflected_enabled()))
	{
		Set_DVGA_Attenuation_Reflected(intra_system_number, config->get_default_DVGA_attenuation_reflected());
		Set_DVGA_Amplifier_Reflected_Enable(intra_system_number, config->get_default_DVGA_amplifier_reflected_enabled());
	}
	else if (ALL_enable == true && DVGA_reflected_ALL_enable_log == false)	//went from ALL disabled to enabled
	{
		Set_DVGA_Attenuation_Reflected(intra_system_number, target_DVGA_attenuation_reflected);
		Set_DVGA_Amplifier_Reflected_Enable(intra_system_number, target_DVGA_amplifier_reflected_enable);
		DVGA_reflected_ALL_enable_log = ALL_enable;
	}
}

/**********************************************************************************************************************************************************************************
 * PWM : set / get the state and configuration of PWM.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_PWM_enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	QString rx = "";
	double temp_target = 100;

	qDebug() << "set_PWM_enable(" << intrasys_num << "):" << enable;

	if (enable == true)
	{
		temp_target = PWM_duty_cycle_target;
	}

	rx = SGX_port->writeRead("$DCS," + QString::number(channel_number) + "," + QString::number(temp_target));

	if (checkStringContents(rx, "$DCS,", ",OK\r\n", ",ERR"))
	{
		PWM_enable = enable;
		PWM_duty_cycle = temp_target;
	}

	emit signal_PWM_settings_get(intra_system_number, PWM_frequency, PWM_duty_cycle);	//emit that the update has occurred
	poll_CW_enable();
}

void RF_Channel::Set_PWM_Duty_Cycle(int intrasys_num, int duty_cycle)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (duty_cycle >= 1 && duty_cycle <= 99)
	{

		Calculate_PWM_Duty_Cycle_Minimum();
		if (duty_cycle < PWM_duty_cycle_minimum)
		{
			if (config->get_use_minimum_pulse_length() == true)
			{
				emit signal_notification(intra_system_number, "Duty cycle must equal or exceed " + QString::number(PWM_duty_cycle_minimum) + "% at " + QString::number(PWM_frequency) + "Hz PWM frequency.");
			}
			duty_cycle = PWM_duty_cycle_minimum;
		}

		qDebug() << "set_PWM_Duty_Cycle(" << intrasys_num << "):" << duty_cycle;
		QString rx = SGX_port->writeRead("$DCS," + QString::number(channel_number) + "," + QString::number(duty_cycle));

		if (checkStringContents(rx, "$DCS,", ",OK\r\n", ",ERR"))
		{
			PWM_duty_cycle_target = duty_cycle;
			PWM_duty_cycle = duty_cycle;
			PWM_enable = true;
			emit signal_PWM_settings_get(intra_system_number, PWM_frequency, PWM_duty_cycle_target);	//emit that the update has occurred
			emit signal_CW_enable_get(intra_system_number, !(ALL_enable || DLL_enable || PWM_enable));
		}
	}
}

void RF_Channel::Set_PWM_Duty_Cycle_Target(int intrasys_num, int duty_cycle)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (PWM_enable == true)
	{
		Set_PWM_Duty_Cycle(intrasys_num, duty_cycle);
	}
	else
	{
//		Calculate_PWM_Duty_Cycle_Minimum();
//		if (duty_cycle < PWM_duty_cycle_minimum)
//		{
//			emit signal_notification(intra_system_number, "Duty cycle must equal or exceed " + QString::number(PWM_duty_cycle_minimum) + "% at " + QString::number(PWM_frequency) + "Hz PWM frequency.");
//			duty_cycle = PWM_duty_cycle_minimum;
//		}

		if (duty_cycle >= 1 && duty_cycle <= 99)
		{
			PWM_duty_cycle_target = duty_cycle;
//			emit signal_PWM_settings_get(intra_system_number, PWM_frequency, PWM_duty_cycle_target);	//emit that the update has occurred
		}
	}
}


void RF_Channel::Set_PWM_Frequency(int intrasys_num, double frequency)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (frequency <= config->get_PWM_frequency_max() && frequency >= config->get_PWM_frequency_min())		//Don't let PWM frequency exceed 19800Hz -> forces 100% DC!
	{
		qDebug() << "set_PWM_Frequency(" << intrasys_num << "):" << QString::number(frequency);
		QString rx = SGX_port->writeRead("$DCFS," + QString::number(channel_number) + "," + QString::number(frequency) + ",0,0");

		if (checkStringContents(rx, "$DCFS,", ",OK\r\n", ",ERR"))
		{
			PWM_frequency = frequency;
			emit signal_PWM_settings_get(intra_system_number, PWM_frequency, PWM_duty_cycle);	//emit that the update has occurred
		}

		//Refresh the duty-cycle value and automatically raise the value if it's below the minimum DC cap for the frequency.
		Set_PWM_Duty_Cycle(intra_system_number, PWM_duty_cycle);
	}
	else
	{
		emit signal_notification(intra_system_number, "PWM frequency must be a value between " + QString::number(config->get_PWM_frequency_min()) + " and " + QString::number(config->get_PWM_frequency_max()));
	}
}

void RF_Channel::Get_PWM_Settings()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$DCG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$DCG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");

		/* DCG includes a ton of junk for setting up PWM triggering, but it was never in a functional state, so it is ignored for the time being */
		PWM_frequency = QString(get_list.at(2)).toDouble();
		PWM_triggering_mode = QString(get_list.at(4)).toInt();
		PWM_triggering_slave_port = QString(get_list.at(5)).toInt();
		PWM_triggering_slave_pin = QString(get_list.at(6)).toInt();
		PWM_triggering_master_port = QString(get_list.at(7)).toInt();
		PWM_triggering_master_pin = QString(get_list.at(8)).toInt();
		PWM_duty_cycle = QString(get_list.at(10)).toDouble();

		if (PWM_duty_cycle == 0 || PWM_duty_cycle == 100)
		{
			PWM_enable = false;
		}
		else if (PWM_duty_cycle >= 1 && PWM_duty_cycle <= 99)
		{
			PWM_enable = true;
		}

		//Keep PWM values within software's acceptable boundaries!
		//PWM_frequency should never exceeds 19800Hz, because it forces the minimum PWM duty cycle to 100%

		if (PWM_frequency > config->get_PWM_frequency_max())
		{
			PWM_frequency = config->get_PWM_frequency_max();
			Set_PWM_Frequency(intra_system_number, PWM_frequency);
		}
		if (PWM_frequency < config->get_PWM_frequency_min())
		{
			PWM_frequency = config->get_PWM_frequency_min();
			Set_PWM_Frequency(intra_system_number, PWM_frequency);
		}

		if (PWM_enable == true)
		{
			Calculate_PWM_Duty_Cycle_Minimum();
			if (PWM_duty_cycle < PWM_duty_cycle_minimum)
			{
				if (config->get_use_minimum_pulse_length() == true)
				{
					emit signal_notification(intra_system_number, "Duty cycle must equal or exceed " + QString::number(PWM_duty_cycle_minimum) + "% at " + QString::number(PWM_frequency) + "Hz PWM frequency.");
				}
				PWM_duty_cycle = PWM_duty_cycle_minimum;
				Set_PWM_Duty_Cycle(intra_system_number, PWM_duty_cycle);	//Refresh the duty-cycle value and automatically raise the value if it's below the minimum DC cap for the frequency.
			}
		}

//		qDebug() << PWM_enable << PWM_duty_cycle << PWM_duty_cycle_minimum;
	}
}

/* If use of the minimum pulse length is enabled, calculates a lower cap for the duty cycle to ensure accurate measurements for the auto-gain algorithm.
 * Otherwise, simply returns 1% as the minimum. (For Feed Forward implementations) */
void RF_Channel::Calculate_PWM_Duty_Cycle_Minimum()
{
	if (config->get_use_minimum_pulse_length() == true)
	{
		PWM_duty_cycle_minimum = PWM_frequency * config->get_PWM_minimum_pulse_length_ns() / 10000000;
		double leftover = fmod(PWM_duty_cycle_minimum, 1.0);

		if (leftover > 0 && leftover < 0.5){
			PWM_duty_cycle_minimum = round(PWM_duty_cycle_minimum + 1);
		}
		else
		{
			PWM_duty_cycle_minimum = round(PWM_duty_cycle_minimum);
		}
	}
	else
	{
		PWM_duty_cycle_minimum = 1;
	}
}


/**********************************************************************************************************************************************************************************
 * SWEEP : set variables for sweeping and perform a sweep.
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_SWP_Power_Watt(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_SWP_Power_Watt(" << intrasys_num << ") -> input value as string: " << QString::number(val, 'f', 5);

	/* check if software needs to sanitize inputs */
	if (config->get_enforce_software_limits_power())
	{
		/* Determine the maximum sweep power value - There is a hard limit of 500W on the firmware */
		double swp_watt_max = 500;
		if (config->get_power_watt_max() < swp_watt_max)
		{
			swp_watt_max = config->get_power_watt_max();
		}

		qDebug() << "Sweep power setpoint must be between" << config->get_power_watt_min() << "and" << swp_watt_max << "watt.";

		/* Either auto-correct the input to the defined limits or deny the input altogether */
		if (config->get_enforce_software_limits_autocorrect())
		{
			if (val > swp_watt_max){
				val = swp_watt_max;
			}
			else if (val < config->get_power_watt_min()){
				val = config->get_power_watt_min();
			}
		}
		else if (!(val <= config->get_power_watt_max() && val >= config->get_power_watt_min()))
		{
			emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
			return;
		}
	}

	//
	// TODO:
	// Hard-coded limit to prevent software from breaking.
	// if power is less than 0.001 watt --> Numpad gets stuck at negative dBm input.
	// if power is 0W --> power in dBm becomes infinitely small
	//
	if (val < 0.001)
	{
		val = 0.001;
	}
	else if (val > 500)
	{
		val = 500;
	}

	SWP_power_watt = val;
	SWP_power_dbm = convert_watt_to_dbm(val);
	emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
}

void RF_Channel::Set_SWP_Power_dBm(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_SWP_Power_dBm(" << intrasys_num << ") -> input value as string: " << QString::number(val, 'f', 5);

	/* check if software needs to sanitize inputs */
	if (config->get_enforce_software_limits_power())
	{
		/* Determine the maximum sweep power value - There is a hard limit of 500W on the firmware */
		double swp_dBm_max = convert_watt_to_dbm(500);
		if (convert_watt_to_dbm(config->get_power_watt_max()) < swp_dBm_max)
		{
			swp_dBm_max = convert_watt_to_dbm(config->get_power_watt_max());
		}

		qDebug() << "power setpoint must be between" << convert_watt_to_dbm(config->get_power_watt_min()) << "and" << convert_watt_to_dbm(config->get_power_watt_max()) << "dBm.";

		/* Either auto-correct the input to the defined limits or deny the input altogether */
		if (config->get_enforce_software_limits_autocorrect())
		{
			if (val > convert_watt_to_dbm(config->get_power_watt_max())){
				val = convert_watt_to_dbm(config->get_power_watt_max());
			}
			else if (val < convert_watt_to_dbm(config->get_power_watt_min())){
				val = convert_watt_to_dbm(config->get_power_watt_min());
			}
		}
		else if (!(val <= convert_watt_to_dbm(config->get_power_watt_max()) && val >= convert_watt_to_dbm(config->get_power_watt_min())))
		{
			emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
			return;
		}
	}

	//
	// TODO:
	// Hard-coded limit to prevent software from breaking.
	// if power is less than 0 dBm --> Numpad gets stuck at negative dBm input.
	// if power is less than -10 dBm --> Power in watt becomes 0W, which is not supported
	//
	if (val < 0)
	{
		val = 0;
	}
	else if (val > convert_watt_to_dbm(500))
	{
		val = convert_watt_to_dbm(500);
	}

	SWP_power_dbm = val;
	SWP_power_watt = convert_dbm_to_watt(val);
	emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
}

void RF_Channel::Set_SWP_Frequency_Start(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_SWP_Frequency_Start(" << intrasys_num << ") -> input value as string: " << QString::number(val, 'f', 5);

	if (config->get_enforce_software_limits_frequency() == true)
	{
		/* Either auto-correct the input to the defined limits or deny the input altogether */
		if (config->get_enforce_software_limits_autocorrect() == true)
		{
			if (val < config->get_frequency_min()){
				val = config->get_frequency_min();
			}
			else if (val >= SWP_frequency_stop){
				val = SWP_frequency_stop - 1;		//Don't set lower frequency higher than upper frequency
			}
		}
		else if (!(val <= (SWP_frequency_stop - 1) && val >= config->get_frequency_min()))
		{
			emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
			return;
		}
	}

	SWP_frequency_start = val;

	/* Adjust step frequency as needed according to the new start/stop values. */
	Set_SWP_Frequency_Step(intra_system_number, SWP_frequency_step);

	emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
}

void RF_Channel::Set_SWP_Frequency_Stop(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_SWP_Frequency_Stop(" << intrasys_num << ") -> input value as string: " << QString::number(val, 'f', 5);

	/* check if software needs to sanitize inputs */
	if (config->get_enforce_software_limits_frequency() == true)
	{
		if (config->get_enforce_software_limits_autocorrect() == true)
		{
			if (val > config->get_frequency_max()){
				val = config->get_frequency_max();
			}
			else if (val <= SWP_frequency_start){
				val = SWP_frequency_start + 1;	//Avoid issues if start freq = 2400 MHz
			}
		}
		else if (!(val <= config->get_frequency_max() && val >= (SWP_frequency_start + 1)))
		{
			emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
			return;
		}
	}

	SWP_frequency_stop = val;

	/* Adjust step frequency as needed according to the new start/stop values. */
	Set_SWP_Frequency_Step(intra_system_number, SWP_frequency_step);

	emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
}

void RF_Channel::Set_SWP_Frequency_Step(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_SWP_Frequency_Step(" << intrasys_num << ") -> input value as string: " << QString::number(val, 'f', 5);
	double sweep_range = SWP_frequency_stop - SWP_frequency_start;
	double minimum_sweep_step = 0.01;

	//
	// TODO:
	// Sweep can use smaller step values, but it only reports 2 digits which prevents the graphing function from doing its job properly cause all the samples end up on the same point...
	// Sweep requires a higher precision in the firmware, then this part of the code can be removed.
	//

	/* Determine the smallest step that should be used */
	if ((sweep_range / (config->get_SWP_maximum_measure_points() - 1)) > minimum_sweep_step)
	{
		minimum_sweep_step = (sweep_range / (config->get_SWP_maximum_measure_points() - 1));
	}

	/* It's inconsistent with the rest of the behaviours, but leaving a potentially incorrect sweep step value may cause things to not play very nice.
	 * So the Sweep step will always be corrected automatically. */
//	if (config->get_enforce_software_limits_frequency() == true)
//	{
//		if (config->get_enforce_software_limits_autocorrect() == true)
//		{
			if (val > sweep_range)
			{
				val = sweep_range;
			}
			else if (val < minimum_sweep_step)
			{
				val = minimum_sweep_step;
			}
//		}
//		else if (!(val >= minimum_sweep_step && val <= sweep_range))
//		{
//			emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
//			return;
//		}
//	}

	SWP_frequency_step = val;
	emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
}

/* Sweep is an irregular feature that returns a lot of lines */
void RF_Channel::Execute_SWP_dBm(int intrasys_num)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Running Sweep dBm(" << intra_system_number << "):" << SWP_frequency_start << SWP_frequency_stop << SWP_frequency_step << SWP_power_dbm;
	QString tx = "$SWPD," + QString::number(channel_number)
				 + "," + QString::number(SWP_frequency_start)
				 + "," + QString::number(SWP_frequency_stop)
				 + "," + QString::number(SWP_frequency_step)
				 + "," + QString::number(SWP_power_dbm) + ",0\r\n";

	QString rx = SGX_port->writeReadOK(tx, 30000);

	if (checkStringContents(rx, "$SWPD,", ",OK\r\n", ",ERR"))
	{
		//
		// TODO:
		// Currently the filtering for bad data is occuring in the SWP handler function
		// That should probably be here instead
		//
	}

	SWP_raw_data = rx;
	emit signal_SWP_measurements_get(intra_system_number, SWP_raw_data);
}

/* Execute_SWP_dBm2: Sweep function where all arguments are provided in one go - For Modbus */
void RF_Channel::Execute_SWP_dBm2(int intrasys_num, double freq_start, double freq_stop, double freq_step, double pow_dbm)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	//Update GUI parameters
	Set_SWP_Frequency_Start(intra_system_number, freq_start);
	Set_SWP_Frequency_Stop(intra_system_number, freq_stop);
	Set_SWP_Frequency_Step(intra_system_number, freq_step);
	Set_SWP_Power_dBm(intra_system_number, pow_dbm);

	qDebug() << "Running Sweep dBm(" << intra_system_number << "):" << SWP_frequency_start << SWP_frequency_stop << SWP_frequency_step << SWP_power_dbm;

	QString tx = "$SWPD," + QString::number(channel_number)
				 + "," + QString::number(SWP_frequency_start)
				 + "," + QString::number(SWP_frequency_stop)
				 + "," + QString::number(SWP_frequency_step)
				 + "," + QString::number(SWP_power_dbm) + ",0\r\n";

	QString rx = SGX_port->writeReadOK(tx, 30000);

	if (checkStringContents(rx, "$SWPD,", ",OK\r\n", ",ERR"))
	{
		//
		// TODO:
		// Currently the filtering for bad data is occuring in the SWP handler function
		// That should probably be here instead
		//
	}

	SWP_raw_data = rx;
	emit signal_SWP_measurements_get(intra_system_number, SWP_raw_data);
}


/**********************************************************************************************************************************************************************************
 * PID : set variables for a PID controller
 *********************************************************************************************************************************************************************************/
void RF_Channel::Set_PID_Kp(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	PID_kp = val;
	emit signal_PID_settings_get(intra_system_number, PID_kp, PID_ki, PID_kd, PID_setpoint, PID_scaling, PID_offset);
}

void RF_Channel::Set_PID_Ki(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	PID_ki = val;
	emit signal_PID_settings_get(intra_system_number, PID_kp, PID_ki, PID_kd, PID_setpoint, PID_scaling, PID_offset);
}

void RF_Channel::Set_PID_Kd(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	PID_kd = val;
	emit signal_PID_settings_get(intra_system_number, PID_kp, PID_ki, PID_kd, PID_setpoint, PID_scaling, PID_offset);
}


void RF_Channel::Set_PID_Scaling(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	PID_scaling = val;
	emit signal_PID_settings_get(intra_system_number, PID_kp, PID_ki, PID_kd, PID_setpoint, PID_scaling, PID_offset);
}

void RF_Channel::Set_PID_Offset(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	PID_offset = val;
	emit signal_PID_settings_get(intra_system_number, PID_kp, PID_ki, PID_kd, PID_setpoint, PID_scaling, PID_offset);
}


void RF_Channel::Set_PID_Setpoint(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	PID_setpoint = val;
	emit signal_PID_settings_get(intra_system_number, PID_kp, PID_ki, PID_kd, PID_setpoint, PID_scaling, PID_offset);
}

void RF_Channel::Execute_PID_Loop(int intrasys_num, double PID_process_variable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	if (!RF_Enable())	//PID loop shouldn't attempt to control anything if RF isn't enabled.
	{
		return;
	}

	double PID_control_value = 0;
	double PID_current_error = PID_setpoint - PID_process_variable;

	//Kp
	PID_control_value += PID_kp * PID_current_error;

	//Ki
//	PID_integral += PID_current_error * _dt;
//	PID_control_value += PID_ki * PID_integral;

	//Kd
//	double derivative = (PID_current_error - PID_prev_error) / _dt;
//	PID_control_value = PID_kd * derivative;

	//control += PID_ki * Integral(error);
	//control += PID_kd * Derivative(error);
}

/**********************************************************************************************************************************************************************************
 * Power Supplies:
 *********************************************************************************************************************************************************************************/
/* Safely turn ON or OFF power supplies.
 * ZHL PAs have an issue that the I2C bus gets sabotaged if PAs are unpowered.
 * This function ensures, if an RF splitter is available, that the I2C communication to the PAs gets cut off.
 * This prevents negatively affecting communication to other devices that share the same bus. */
void RF_Channel::Set_PSU_Enable_Safely(int intrasys_num, bool enable)
{
	if (enable == true)
	{
		Set_PSU_Enable(intrasys_num, true);
		Set_RF_splitter_channels_I2C_enable(intrasys_num, 0x0f);	//Does nothing if a splitter board is not configured properly
	}
	else
	{
		Set_RF_splitter_channels_I2C_enable(intrasys_num, 0x00);	//Does nothing if a splitter board is not configured properly
		Set_PSU_Enable(intrasys_num, false);
	}
}

void RF_Channel::Set_PSU_Enable(int intrasys_num, bool enable)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_PSU_Enable(" << intra_system_number << "):" << enable;
	QString rx = SGX_port->writeRead("$PSUES," + QString::number(channel_number) + "," + QString::number(enable));

	if (checkStringContents(rx, "$PSUES,", ",OK\r\n", ",ERR"))
	{
		for (int i = 0; i < PSU_count; i++)
		{
			PSU_enable[i] = enable;
		}

		PSU_enable_combined = enable;
		emit signal_PSU_enable_combined_get(intra_system_number, PSU_enable_combined);	//emit update
	}
}

/* Get the shared PSU enable of all PSUs
 * Note: It only polls the first PSU on firmware level */
void RF_Channel::Get_PSU_Enable()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PSUEG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PSUEG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		PSU_enable_combined = (bool)QString(get_list.at(2)).toInt();

		//This is probably not an entirely reasonable assumption, but I don't really have a choice at this time...
//		for (int i = 0; i < PSU_count; i++)
//		{
//			PSU_enable[i] = (bool)QString(get_list.at(2)).toInt();
//		}
	}
}

/* Set the PSU voltage setpoint */
void RF_Channel::Set_PSU_Voltage_Setpoint(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_PSU_Voltage_Setpoint(" << intra_system_number << "):" << val;
	QString rx = SGX_port->writeRead("$PSUVS," + QString::number(channel_number) + "," + QString::number(val));

	if (checkStringContents(rx, "$PSUVS,", ",OK\r\n", ",ERR"))
	{
		PSU_voltage_setpoint = val;
		emit signal_PSU_voltage_setpoint_get(intra_system_number, PSU_voltage_setpoint);
	}
}

/* Get the PSU voltage setpoint */
void RF_Channel::Get_PSU_Voltage_Setpoint()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PSUVG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PSUVG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		PSU_voltage_setpoint = QString(get_list.at(2)).toDouble();
	}
	else
	{
		PSU_voltage_setpoint = -1;
	}
}

/* Set the PSU current limit of all PSUs combined */
void RF_Channel::Set_PSU_Current_Limit(int intrasys_num, double val)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "Set_PSU_Current_Limit(" << intra_system_number << "):" << val;
	QString rx = SGX_port->writeRead("$PSUIS," + QString::number(channel_number) + "," + QString::number(val));

	if (checkStringContents(rx, "$PSUIS,", ",OK\r\n", ",ERR"))
	{
		PSU_current_limit = val;
		emit signal_PSU_current_limit_get(intra_system_number, PSU_current_limit);	//emit update
	}
}

/* Get the PSU current limit of all PSUs combined */
void RF_Channel::Get_PSU_Current_limit()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PSUIG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PSUIG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		PSU_current_limit = QString(get_list.at(2)).toDouble();
	}
	else
	{
		PSU_current_limit = -1;
	}
}

/* Get individual PSU Voltage and Current of all PSU's in one go
 * order is voltage, current, voltage, current, etc */
void RF_Channel::Get_PSU_IU_All_Reading()
{
	if (!channel_working)
	{
		return;
	}

	PSU_total_power = 0;
	QString get_string = SGX_port->writeRead("$PSUDG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PSUDG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");

		for (int i = 0; i < PSU_count; i++)
		{
			PSU_voltage[i] = QString(get_list.at(2+i*2)).toDouble();
			PSU_current[i] = QString(get_list.at(3+i*2)).toDouble();
			PSU_power[i] = PSU_voltage[i] * PSU_current[i];
			PSU_total_power += PSU_voltage[i] * PSU_current[i];
		}

		bool valid_readings = true;
		for (int i = 0; i < PSU_count; i++)
		{
			if (PSU_voltage[i] < 0 || PSU_current[i] < 0)
			{
				valid_readings = false;
				break;
			}
		}

		/* This exists to communicate that despite PSUDG returning values without an ERROR, the values are nonsensical due to noise on the I2C bus */
		emit signal_PSU_readings_valid(intra_system_number, valid_readings);

		//Avoid divide by zero...
		if (PSU_total_power > 0)
		{
			PSU_power_efficiency = PA_power_forward_watt / PSU_total_power * 100;
		}
		else
		{
			PSU_power_efficiency = 0;
		}
	}
}

/* Unused Functions */


/* Get the shared PSU Voltage of all PSUs */
void RF_Channel::Get_PSU_Voltage_Reading()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PSUVRG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PSUVRG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		PSU_voltage_combined = QString(get_list.at(2)).toDouble();

		//This is possibly not a reasonable assumption?
//		for (int i = 0; i < PSU_count; i++)
//		{
//			PSU_voltage[i] = PSU_voltage_combined;
//		}
	}
}

/* Get combined PSU Current of all PSUs */
void RF_Channel::Get_PSU_Current_Reading()
{
	if (!channel_working)
	{
		return;
	}

	QString get_string = SGX_port->writeRead("$PSUIRG," + QString::number(channel_number));

	if (checkStringContents(get_string, "$PSUIRG,", "\r\n", ",ERR"))
	{
		QStringList get_list = get_string.split(",");
		PSU_current_combined = QString(get_list.at(2)).toDouble();

		//This is not a reasonable assumption!
//		for (int i = 0; i < PSU_count; i++)
//		{
//			PSU_current[i] = PSU_current_combined / PSU_count;
//		}
	}
}

void RF_Channel::Calculate_PSU_Dissipation()
{
	PSU_dissipation_watt = PSU_total_power - PA_power_forward_watt + PA_power_reflected_watt;
}

/**********************************************************************************************************************************************************************************
 * Feature Availability detection: Determine whether particular feature should be available or not.
 *********************************************************************************************************************************************************************************/
/* Determine the availability state of Amplifier Mode */
void RF_Channel::determine_availability_amplifier_mode()
{
	/* External Triggering availability factors:
	 * - Support for feature according to config file.
	 * - External Triggering Mode must be disabled -> SGx boards do not support simultanous amplifier mode and external trigger. Perhaps other hardware could but no use case exists at this time.
	 */

	bool available = false;
	bool visible = false;

	bool factor_config_support = config->get_support_power_control_modes() && config->get_support_amplifier_mode();
	bool factor_external_triggering = external_triggering_enable;

	visible = (factor_config_support == true);
	available = (factor_config_support == true &&
				 factor_external_triggering == false);

	emit signal_availability_amplifier_mode(intra_system_number, visible, available);
}

/* Determine the availability state of External Triggering */
void RF_Channel::determine_availability_external_triggering()
{
	/* External Triggering availability factors:
	 * - Support for feature through config file.
	 * - Amplifier mode must be disabled -> SGx boards do not support simultanous amplifier mode and external trigger. Perhaps other hardware could but no use case exists at this time.
	 */

	bool available = false;
	bool visible = false;

	bool factor_config_support = config->get_support_external_triggering_mode();
	bool factor_amplifier_mode = amplifier_mode_enable;

	visible = (factor_config_support == true);
	available = (factor_config_support == true &&
				 factor_amplifier_mode == false);

	emit signal_availability_external_triggering(intra_system_number, visible, available);
}

/* Determine the availability state of PWM */
void RF_Channel::determine_availability_PWM()
{
	/* PWM availability factors:
	 * - Support for feature according to config file
	 * - Amplifier Mode must be disabled -> Can't PWM an external signal, especially if the external signal is also already PWM'd.
	 * - External triggering must be disabled -> PWM and external triggering are mutually exclusive.
	 */

	bool available = false;
	bool visible = false;

	bool factor_config_support = config->get_controls_PWM_enabled();
	bool factor_amplifier_mode = amplifier_mode_enable;
	bool factor_external_triggering = external_triggering_enable;
//	bool factor_external_triggering_synchronization = external_triggering_synchronization_enable;

	visible = (factor_config_support == true);
	available = (factor_config_support == true &&
				 factor_amplifier_mode == false &&
				 factor_external_triggering == false);

//	qDebug() << "Subsystem[" << intra_system_number << "]: PWM availability factors:" << factor_config_support << factor_amplifier_mode << factor_external_triggering;

	emit signal_availability_PWM(intra_system_number, visible, available);
}

void RF_Channel::determine_availability_DLL()
{
	/* DLL availability factors:
	 * - Support for feature according to config file
	 * - External triggering -> External triggering must be disabled -> DLL unavailable because it cannot know the correct timing to measure power for DLL acitivity.
	 * - Amplifier mode -> Amplifier mode must be disabled -> SGx does not generate the RF signal, there is no control over the frequency, therefor DLL is impossible.
	 */

	bool available = false;
	bool visible = false;

	bool factor_config_support = config->get_controls_DLL_enabled();
	bool factor_amplifier_mode = amplifier_mode_enable;
	bool factor_external_triggering = external_triggering_enable;
	bool factor_external_triggering_synchronization = external_triggering_synchronization_enable;

	visible = (factor_config_support == true);
	available = (factor_config_support == true &&
				 factor_amplifier_mode == false &&
				 (factor_external_triggering == false || (factor_external_triggering == true && factor_external_triggering_synchronization == true)));

	emit signal_availability_DLL(intra_system_number, visible, available);
}

void RF_Channel::determine_availability_ALL()
{
	/* ALL availability factors:
	 * - ALL hardware compatibility
	 * - Support for feature according to config file
	 * - External triggering -> External triggering must be disabled -> ALL unavailable because it cannot know the correct timing to measure power for ALL acitivity.
	 * - Amplifier mode -> SGx does not generate the RF signal, there is no control over the frequency, therefor ALL is impossible.
	 */

	bool available = false;
	bool visible = false;

	bool factor_config_support = config->get_controls_ALL_enabled();
	bool factor_ALL_compatible = ALL_compatible;
	bool factor_amplifier_mode = amplifier_mode_enable;
	bool factor_external_triggering = external_triggering_enable;
	bool factor_external_triggering_synchronization = external_triggering_synchronization_enable;

	visible = (factor_config_support == true && factor_ALL_compatible == true);
	available = (factor_config_support == true && factor_ALL_compatible == true &&
				 factor_amplifier_mode == false &&
				 (factor_external_triggering == false || (factor_external_triggering == true && factor_external_triggering_synchronization == true)));

	emit signal_availability_ALL(intra_system_number, visible, available);
}

void RF_Channel::determine_availability_sweep()
{
	/* External Triggering availability factors:
	 * - Support for feature according to config file
	 * - External triggering -> External triggering must be disabled, Otherwise sweep cannot know the correct timing to measure RF power.
	 * - Amplifier Mode -> SGx does not generate the RF signal, there is no control over the frequency, therefor Sweep is impossible
	 */

	bool available = false;
	bool visible = false;

	bool factor_config_support = config->get_menu_sweep_enabled();
	bool factor_amplifier_mode = amplifier_mode_enable;
	bool factor_external_triggering = external_triggering_enable;
	bool factor_external_triggering_synchronization = external_triggering_synchronization_enable;

	visible = (factor_config_support == true &&
			   factor_amplifier_mode == false);
	available = (factor_config_support == true &&
				 factor_amplifier_mode == false &&
				 (factor_external_triggering == false || (factor_external_triggering == true && factor_external_triggering_synchronization == true)));
	emit signal_availability_sweep(intra_system_number, visible, available);
}

void RF_Channel::determine_availability_GCS_controls()
{

}

void RF_Channel::determine_availability_MCS_controls()
{

}

void RF_Channel::determine_availability_PWRS_controls()
{

}

void RF_Channel::determine_availability_PWRSGDS_controls()
{

}

void RF_Channel::determine_availability_PSU_enable_controls()
{
	/* PSU Enable availability factors:
	 * - Support for feature according to config file
	 * - PSU enable should be controlled only by software or only by hardware, but not both.
	 */

	bool available = false;
	bool visible = false;

	bool factor_config_support = config->get_support_PSU_controls_enable();
	bool factor_hardware_enable = config->get_GPIO_interlock_in() <= 0;	//interlock GPIO must not be provided. (interlock GPIO is used as PSU enable trigger)

	visible = (factor_config_support == true);
	available = (factor_config_support == true &&
				 factor_hardware_enable == true);

	emit signal_availability_PSU_enable(intra_system_number, visible, available);
}


/* Determine the LED blink mode */
#if defined (Q_OS_LINUX)
/*
 * Handle LED lighting
 * Mode 0 - RF OFF – LED OFF
 * Mode 1 - RF ON – LED ON (solid)
 * Mode 2 - ALARM - LED Blink (500ms ON -> 500ms OFF)
 * Mode 3 - Blind RCM - LED Blink (1000ms ON -> 1000ms OFF)
 */
void RF_Channel::determine_LED_mode()
{
	/* This function is called by RF_enable and by poll_status whenever the error status changes */
	if (config->get_support_HAT_B14_0835())
	{
		LED_MODE mode = LED_MODE::LED_MODE_OFF;

		//
		// TODO:
		// Blind RCM is not yet supported
		// When blind RCM becomes available the LED should blink slowly when blind RCM is enabled
		//

//		if (RCM_blind_mode != 0)
//		{
//			mode = LED_mode::LED_MODE_RCM;
//		}
//		else if (error != 0)

		if (error != 0)
		{
			mode = LED_MODE::LED_MODE_ALARM;
		}
		else if (RF_enable == true)
		{
			mode = LED_MODE::LED_MODE_ON;
		}
		else if (RF_enable == false)
		{
			mode = LED_MODE::LED_MODE_OFF;
		}

		//emit LED mode
		if (mode != last_LED_mode)
		{
			emit signal_LED_mode_set(intra_system_number, mode);
		}

		last_LED_mode = mode;
	}
}
#endif

/**********************************************************************************************************************************************************************************
 * Channel Polling: periodically request all important data from the channel.
 *********************************************************************************************************************************************************************************/

void RF_Channel::poll_determine_availability_features()
{
	determine_availability_amplifier_mode();
	determine_availability_external_triggering();
	determine_availability_PWM();
	determine_availability_ALL();
	determine_availability_DLL();
	determine_availability_sweep();
	determine_availability_PSU_enable_controls();
//	determine_availability_GCS_controls();
//	determine_availability_MCS_controls();
//	determine_availability_PWRS_controls();
//	determine_availability_PWRSGDS_controls();
}

void RF_Channel::poll_RF_enable()
{
	Get_RF_enable();
	emit signal_RF_enable_get(intra_system_number, RF_enable);

	//In case the RF enable state changes (for example via RCM) the LED mode should change too.
#if defined (Q_OS_LINUX)
	if (config->get_support_HAT_B14_0835())
	{
		determine_LED_mode();
	}
#endif
}

void RF_Channel::poll_power_dbm()
{
	Get_Power_dBm();
	emit signal_power_get(intra_system_number, power_dbm, power_watt);
}

void RF_Channel::poll_frequency()
{
	Get_Frequency();
	emit signal_frequency_get(intra_system_number, frequency_mhz);
}

void RF_Channel::poll_offset_db()
{
	Get_Offset_dB();
	emit signal_offset_db_get(intra_system_number, offset_db);
}

/* Poll the attenuation of the SGx board. */
void RF_Channel::poll_VGA_attenuation()
{
	if (power_control_mode != POWER_CONTROL_NORMAL)
	{
		Get_VGA_attenuation();
		emit signal_VGA_attenuation_get(intra_system_number, VGA_attenuation);
	}
}

/* Poll the magnitude of the SGx board. */
void RF_Channel::poll_IQMod_magnitude()
{
	if (power_control_mode != POWER_CONTROL_NORMAL)
	{
		Get_IQMod_Magnitude();
		emit signal_IQMod_magnitude_get(intra_system_number, IQMod_magnitude);
	}
}

/* Poll the phase of the SGx board. */
void RF_Channel::poll_phase()
{
	if (config->get_support_phase_control() == true)
	{
		Get_Phase();
		emit signal_phase_get(intra_system_number, phase_degrees);
	}
}

/* Poll the attenuation of the RF splitter channels. */
void RF_Channel::poll_VGA_attenuation_nchannel()
{
	if (use_substitute_channel_number == true && config->get_support_splitter_controls_attenuation() == true)
	{
		/* Workaround:
		 * There is (currently) no splitter type that supports more than 4 channels.
		 * To prevent the GUI from making requests to channels that do no exist and greying out due to communication failures, this feature is limited to 4channels max. This allows $PPG2/$PPDG2 to continue working for any channel count */

		int workaround_splitter_channel_count = splitter_channel_count;
		if (workaround_splitter_channel_count > 4)
		{
			workaround_splitter_channel_count = 4;
		}

		for (int i = 1; i <= workaround_splitter_channel_count; i++)
		{
			Get_Attenuation_NChannel(i + channel_number - 1);	//use actual sub-channel number
			emit signal_NChannel_attenuation_get(intra_system_number, i, attenuation_nchannel[i-1]);
		}
	}
}

/* Poll the magnitude of the RF splitter channels. */
void RF_Channel::poll_IQMod_magnitude_nchannel()
{
	if (use_substitute_channel_number == true && config->get_support_splitter_controls_magnitude() == true)
	{
		/* Workaround:
		 * There is (currently) no splitter type that supports more than 4 channels.
		 * To prevent the GUI from making requests to channels that do no exist and greying out due to communication failures, this feature is limited to 4channels max. This allows $PPG2/$PPDG2 to continue working for any channel count */

		int workaround_splitter_channel_count = splitter_channel_count;
		if (workaround_splitter_channel_count > 4)
		{
			workaround_splitter_channel_count = 4;
		}

		for (int i = 1; i <= workaround_splitter_channel_count; i++)
		{
			Get_Magnitude_NChannel(i + channel_number - 1);	//use actual sub-channel number
			emit signal_NChannel_magnitude_get(intra_system_number, i, magnitude_nchannel[i-1]);
		}
	}
}

/* Poll the phase of the RF splitter channels. */
void RF_Channel::poll_phase_nchannel()
{
	if (use_substitute_channel_number == true && config->get_support_splitter_controls_phase() == true)
	{
		/* Workaround:
		 * There is (currently) no splitter type that supports more than 4 channels.
		 * To prevent the GUI from making requests to channels that do no exist and greying out due to communication failures, this feature is limited to 4channels max. This allows $PPG2/$PPDG2 to continue working for any channel count */

		int workaround_splitter_channel_count = splitter_channel_count;
		if (workaround_splitter_channel_count > 4)
		{
			workaround_splitter_channel_count = 4;
		}

		for (int i = 1; i <= workaround_splitter_channel_count; i++)
		{
			Get_Phase_NChannel(i + channel_number - 1);	//use actual sub-channel number
			emit signal_NChannel_phase_get(intra_system_number, i, phase_degrees_nchannel[i-1]);
		}
	}
}

void RF_Channel::poll_autophase()
{
	Get_AutoPhase();
	emit signal_autophase_get(intra_system_number, autophase_degrees);
}

void RF_Channel::poll_autophase_enable()
{
	Get_AutoPhase_Enable();
	emit signal_autophase_enable_get(intra_system_number, autophase_enable);
}

void RF_Channel::poll_demodulator_phase()
{
	//
	// TODO:
	// Should sphase polling be limited by the state of the demodulator?
	//
	//$SPhase seems to be nonsense if auto-phase isn't enabled; therefor don't do anything.
	if (demodulator_enable == true)
	{
		Get_Phase_Demodulator();
		emit signal_demodulator_phase_get(intra_system_number, phase_degrees_demodulator);
	}
}

void RF_Channel::poll_demodulator_enable()
{
	Get_Demodulator_Enable();
	emit signal_demodulator_enable_get(intra_system_number, demodulator_enable);
}

void RF_Channel::poll_temperature_PA()
{
	Get_PA_Temperature();
	emit signal_temperature_PA_get(intra_system_number, temperature_PA_average);
}

void RF_Channel::poll_temperature_SG()
{
	Get_SG_Temperature();
	emit signal_temperature_SG_get(intra_system_number, temperature_SG_average);
}

/* Check if error state has changed, Emit signal appropriate signal and information if that's the case. */
void RF_Channel::poll_SG_error()
{
	if (!channel_working)
	{
		return;
	}

	last_error = error;
	Get_Status();

	// Error state has changed;
	if (last_error != error)
	{
		emit signal_error_get(intra_system_number, error, statuschecker->translate_SG_error(error));

#if defined (Q_OS_LINUX)
		if (config->get_support_HAT_B14_0835())
		{
			determine_LED_mode();
		}
#endif
	}

	/* If there's an error, start polling the RF enable state to see if SOA is shutting down the RF power */
	if (error > 0)
	{
		//if last error had no reset error, but new one does
		if ((error & (1<<5)) && !(last_error & (1<<5)))
		{
			emit signal_reset_detected(intra_system_number, channel_number);
		}

		if (error_clearing_enable == true)
		{
			Execute_Error_Clear(intra_system_number);
		}
	}

	/* Also check for resets based on the $RTG run time value */
	if (config->get_use_runtime_based_reset_detection() == true)
	{
		Detect_Runtime_SGx_Reset();
	}
}

void RF_Channel::poll_external_triggering_mode()
{
	Get_External_Triggering_Synchronization_Enable();
//	Get_External_Triggering_Synchronization_Delay();
	Get_External_Triggering_Enable();

	determine_availability_external_triggering();

	emit signal_external_triggering_synchronization_enable_get(intra_system_number, external_triggering_synchronization_enable);
//	emit signal_external_triggering_synchronization_delay_get(intra_system_number, external_triggering_synchronization_delay_us);
	emit signal_external_triggering_enable_get(intra_system_number, external_triggering_enable);
}

void RF_Channel::poll_power_control_mode()
{
	//Get_PID_Control_Enable();
	//Get_Analog_Input_Enable();
	Get_Amplifier_Mode_Enable();
	Get_Autogain_Enable();

	/* Update the power control mode */
	Identify_Power_Control_Mode();
	if (power_control_mode != last_power_control_mode)
	{
		emit signal_power_control_mode_get(intra_system_number, power_control_mode);
	}
	last_power_control_mode = power_control_mode;
}

void RF_Channel::poll_clock_source()
{
	Get_Clock_Source();
	emit signal_clock_source_get(intra_system_number, clock_source);
}

void RF_Channel::poll_CW_enable()
{
	CW_enable = (ALL_enable == false && DLL_enable == false && PWM_enable == false);
	emit signal_CW_enable_get(intra_system_number, CW_enable);
}

void RF_Channel::poll_DLL_settings()
{
	Get_DLL_Settings();
	emit signal_DLL_settings_get(intra_system_number, DLL_frequency_limit_lower, DLL_frequency_limit_upper, DLL_start_frequency, DLL_step_frequency, DLL_threshold, DLL_main_delay);
}

void RF_Channel::poll_DLL_enable()
{
	Get_DLL_enable();
	emit signal_DLL_enable_get(intra_system_number, DLL_enable);
}

void RF_Channel::poll_ALL_settings()
{
	Get_ALL_Settings();
	emit signal_ALL_settings_get(intra_system_number, ALL_frequency_limit_lower, ALL_frequency_limit_upper, ALL_threshold);
}

void RF_Channel::poll_ALL_enable()
{
	Get_ALL_enable();
	emit signal_ALL_enable_get(intra_system_number, ALL_enable);
}

void RF_Channel::poll_PWM_settings()
{
	Get_PWM_Settings();
	emit signal_PWM_settings_get(intra_system_number, PWM_frequency, PWM_duty_cycle);
	emit signal_PWM_triggering_get(intra_system_number, PWM_triggering_mode);
}

void RF_Channel::poll_DVGA_forward_settings()
{
	Get_DVGA_Forward_Settings();
	emit signal_DVGA_forward_settings_get(intra_system_number, DVGA_amplifier_forward_enable, DVGA_attenuation_forward);
}

void RF_Channel::poll_DVGA_reflected_settings()
{
	Get_DVGA_Reflected_Settings();
	emit signal_DVGA_reflected_settings_get(intra_system_number, DVGA_amplifier_reflected_enable, DVGA_attenuation_reflected);
}

void RF_Channel::poll_SWP_settings()
{
	//There's nothing to poll, just periodically relay the stored variables to the GUI...
	emit signal_SWP_settings_get(intra_system_number, SWP_frequency_start, SWP_frequency_stop, SWP_frequency_step, SWP_power_dbm, SWP_power_watt);
}

void RF_Channel::poll_PID_settings()
{
	emit signal_PID_settings_get(intra_system_number, PID_kp, PID_ki, PID_kd, PID_setpoint, PID_scaling, PID_offset);
}

void RF_Channel::poll_PSU_enable()
{
	Get_PSU_Enable();
	emit signal_PSU_enable_combined_get(intra_system_number, PSU_enable_combined);
}

void RF_Channel::poll_PSU_IU_All_Readings()
{
	Get_PSU_IU_All_Reading();
	for (int i = 0; i < PSU_count; i++)
	{
		emit signal_PSU_IU_reading_get(intra_system_number, i, PSU_voltage.at(i), PSU_current.at(i), PSU_power.at(i));
	}
	emit signal_PSU_power_efficiency_get(intra_system_number, PSU_power_efficiency);
}

void RF_Channel::poll_PSU_voltage_setpoint()
{
	Get_PSU_Voltage_Setpoint();
	emit signal_PSU_voltage_setpoint_get(intra_system_number, PSU_voltage_setpoint);
}

void RF_Channel::poll_PSU_current_limit()
{
	Get_PSU_Current_limit();
	emit signal_PSU_current_limit_get(intra_system_number, PSU_current_limit);
}

/* Unused polls */
void RF_Channel::poll_PSU_voltage()
{
	Get_PSU_Voltage_Reading();
	emit signal_PSU_voltage_get(intra_system_number, PSU_voltage_combined);
}

void RF_Channel::poll_PSU_current()
{
	Get_PSU_Current_Reading();
	emit signal_PSU_current_get(intra_system_number, PSU_current_combined);
}

void RF_Channel::poll_PSU_dissipation()
{
	Calculate_PSU_Dissipation();
	emit signal_PSU_dissipation_get(intra_system_number, PSU_dissipation_watt);
}
/**********************************************************************************************************************************************************************************
 * Remote Command Mode: Exchange messages between RCM port and SGX port.
 *********************************************************************************************************************************************************************************/
//
// TODO:
// This prevents the use of $PCS,2 $PCS,3 $PCS,4 needed to configure the different channels of Phase Gain Splitter Boards.
// The implementation of phase gain splitter boards is currently incompatible with RCM.
//

/* LIVE RCM mode sends messages without having to halt the regular operations of the RF channel.
 * As a trade-off it can only deal with single-line responses. */
void RF_Channel::RCM_Live_serial_writeRead_to_SGx(QString message)
{
	if (!channel_working)
	{
		return;
	}

	/* Ensure this channel is spoken to. */
	QRegExp regexp("\\$\\w{1,},(\\d+)");
	regexp.indexIn(message);

	bool permit_communication = false;
	if (regexp.cap(1) == "0" || regexp.cap(1) == QString::number(channel_number))
	{
		permit_communication = true;
	}

	if (permit_communication == false && use_substitute_channel_number == true)
	{
		if (regexp.cap(1) == "200" || regexp.cap(1) == QString::number(channel_number + 200))
		{
			permit_communication = true;
		}
		for (int i = 1; i < splitter_channel_count; i++)
		{
			if (regexp.cap(1) == QString::number(channel_number + i))
			{
				permit_communication = true;
				break;
			}
		}
	}

	if (permit_communication == true)
	{
		QString response = SGX_port->writeRead(message);	//May behave a bit derpy when a no response is emitted by the SGX_port...
		qDebug() << "RCM RX:\t" << response << "\n";
		emit signal_RCM_serial_response(response);
	}
}

void RF_Channel::RCM_Live_serial_Sweep_handler(QString message)
{
	if (!channel_working)
	{
		return;
	}

	/* Ensure this channel is spoken to. */
	QRegExp regexp("\\$\\w{1,},(\\d+)");
	regexp.indexIn(message);

	bool permit_communication = false;
	if (regexp.cap(1) == "0" || regexp.cap(1) == QString::number(channel_number))
	{
		permit_communication = true;
	}

	if (permit_communication == false && use_substitute_channel_number == true)
	{
		if (regexp.cap(1) == "200" || regexp.cap(1) == QString::number(channel_number + 200))
		{
			permit_communication = true;
		}
		for (int i = 1; i < splitter_channel_count; i++)
		{
			if (regexp.cap(1) == QString::number(channel_number + i))
			{
				permit_communication = true;
				break;
			}
		}
	}

	if (permit_communication == true)
	{
		QString rx = SGX_port->writeReadOK(message, 30000);

		qDebug() << "RCM RX:\t" << rx << "\n";
		emit signal_RCM_serial_response(rx);
	}
}

/* Blind RCM mode
 * Cannot be used while other readings are happening because the user would get spammed.
 * No limitations on the amount of lines to send and receive, as everything is handled immediately as it comes */
void RF_Channel::RCM_Blind_serial_write_to_SGx(QString message)
{
	SGX_port->RCM_Write(message);
}

void RF_Channel::RCM_Blind_serial_handleResponse_from_SGx()
{
	emit signal_RCM_serial_response(SGX_port->RCM_Read());
}


/* TEST function of the firmware. Functionality prone to change any time. */
void RF_Channel::poll_TEST()
{
	TEST(intra_system_number,63);
}

void RF_Channel::TEST(int intrasys_num, int count)
{
	if (!channel_working || intrasys_num != intra_system_number)
	{
		return;
	}

	qDebug() << "TEST [" << intrasys_num << "]:" << count;
	QString rx = SGX_port->writeRead("$TST," + QString::number(channel_number) + "," + QString::number(count));
//	qDebug() << rx;
}
